/**
 * Copyright 2020 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @fileoverview Express web server for Little Bear Cache sample
 *
 * The /echo endpoint accepts POST requests and forwards the request body on to
 * Google Standard Payments' `echo` endpoint.
 *
 * The /v1/paycache/echo endpoint handles inbound echo requests from Google
 * Standard Payments and responds successfully.
 *
 */

const express = require('express');
const base64url = require('base64url');
const axios = require('axios');
const crypto = require('./crypto.js');
const bodyHelpers = require('./bodyHelpers.js');
const bodyParser = require('body-parser');

const VERBOSE = true; // Enable verbose logging in the echo tests.
const PIAID = 'YOUR_PIAID_HERE_1';

const HOST = 'billpaynotification.sandbox.googleapis.com';
const ECHO_URL = 'https://' + HOST + '/secure-serving/gsp/v1/echo/' + PIAID;
const AXIOS_CONFIG = {
    headers: { 'Content-Type': 'application/octet-stream' },
};

const app = express();
app.set('trust proxy', true);
const rawParser = bodyParser.raw();
const textParser = bodyParser.text();
app.use('/public', express.static(__dirname + '/public'));

// Listen to the App Engine-specified port, or 8080 otherwise
const PORT = process.env.PORT || 8080;
app.listen(PORT, () => {
    console.log(`Server listening on port ${PORT}...`);
});

app.post('/v1/paycache/echo', rawParser, async (req, res) => {
    const encryptedRequest = base64url.toBuffer(req.body);
    const decryptedRequest = await crypto.decrypt(encryptedRequest);
    if (VERBOSE) { console.log('Decrypted request:\n', decryptedRequest); }
    clientMessage = JSON.parse(decryptedRequest).clientMessage;
    responseBody = bodyHelpers.buildEchoResponseBody(clientMessage);
    if (VERBOSE) { console.log('Response body:\n', responseBody); }
    encryptedResponse = await crypto.encrypt(responseBody);
    if (VERBOSE) { console.log('Encrypted response:\n', encryptedResponse); }
    const encodedResponse = base64url(encryptedResponse);
    if (VERBOSE) { console.log('Encoded response:\n', encodedResponse); }
    res.send(encodedResponse);
});

// Accept POST requests to /echo, forwarding the request body to GSP's `echo` endpoint
app.post('/echo', textParser, async (req, res) => {
    if (VERBOSE) { console.log('Content-Type:\n', req.headers['content-type']); }
    if (VERBOSE) { console.log('Request body:\n', req.body); }
    const message = bodyHelpers.buildEchoRequestBody(req.body);
    const encrypted = await crypto.encrypt(message);
    if (VERBOSE) { console.log('Encrypted message:\n', encrypted); }
    try {
        const response =
            await axios.post(ECHO_URL, base64url(encrypted), AXIOS_CONFIG);
        const encryptedMessage = base64url.toBuffer(response.data);
        const decryptedResponse = await crypto.decrypt(encryptedMessage);
        if (VERBOSE) { console.log('Decrypted response: ', decryptedResponse); }
        res.status(200);
        res.send(decryptedResponse);
    } catch (error) {
        if (VERBOSE) { console.log(error); }
        res.status(500);
        res.send(error);
    }
});