﻿using System;
using System.IO;
using Google.Apis.RCSBusinessMessaging.v1.Data;

using Google.Apis.Auth.OAuth2;
using Google.Cloud.PubSub.V1;
using Grpc.Auth;
using Grpc.Core;

using System.Linq;
using System.Text;
using System.Threading;
using System.Collections.Generic;

using RCSBusinessMessaging;

using Newtonsoft.Json;
using Newtonsoft.Json.Linq;

namespace rbm_csharp_client_v1
{
    /// <summary>
    /// RCS Business Messaging sample first agent.
    ///
    /// Sends the following message to a user: "What is your favorite color?"
    /// Parses the user's response and echos it in a new message.
    /// </summary>
    class FirstAgent
    {
        private string subscriptionId = "rbm-agent-subscription";

        // the phone number, in E.164 format, to start a conversation with
        private string msisdn;

        // RBM api helper, makes SDK easier to use
        private RbmApiHelper rbmApiHelper;

        /// <summary>
        /// Initializes a new instance of the <see cref="T:rbm_csharp_client_v1.FirstAgent"/> class.
        /// </summary>
        /// <param name="credentialsFileLocation">The service credentials file location.</param>
        /// <param name="projectId">The Google Cloude project identifier.</param>
        /// <param name="msisdn">The client msisdn in E.164 format.</param>
        public FirstAgent(string credentialsFileLocation,
                          string projectId, string msisdn)
        {
            this.rbmApiHelper = new RbmApiHelper(credentialsFileLocation,
                                                 projectId);

            this.msisdn = msisdn;

            InitPullMessages(projectId, credentialsFileLocation);
        }


        /// <summary>
        /// Inits the Pub/Sub subscription handler.
        /// </summary>
        /// <param name="projectId">The GCP project identifier.</param>
        /// <param name="jsonPath">The service account credentials file location.</param>
        private async void InitPullMessages(string projectId, string jsonPath)
        {
            GoogleCredential googleCredential = null;
            using (var jsonStream = new FileStream(jsonPath, FileMode.Open,
                FileAccess.Read, FileShare.Read))
            {
                googleCredential = GoogleCredential.FromStream(jsonStream)
                    .CreateScoped(SubscriberServiceApiClient.DefaultScopes);
            }

            SubscriptionName subscriptionName = new SubscriptionName(projectId, subscriptionId);
            SubscriberClient subscriber = new SubscriberClientBuilder
            {
                SubscriptionName = subscriptionName,
                GoogleCredential = googleCredential

            }.Build();
            
            // setup listener for pubsub messages
            await subscriber.StartAsync(
                async (PubsubMessage message, CancellationToken cancel) =>
                {
                    string text =
                        System.Text.Encoding.UTF8.GetString(message.Data.ToArray());

                    var jsonObject = JsonConvert.DeserializeObject<IDictionary<string, object>>(text);

                    // check if the message is a user response message
                    if (jsonObject.ContainsKey("text"))
                    {
                        string messageId = (string)jsonObject["messageId"];
                        string userResponse = (string)jsonObject["text"];
                        string msisdn = (string)jsonObject["senderPhoneNumber"];

                        // let the user know their message has been read
                        rbmApiHelper.SendReadMessage(messageId, msisdn);

                        HandleUserResponse(userResponse, msisdn);
                    }

                    await Console.Out.WriteLineAsync(
                        $"Message {message.MessageId}: {text}");
                    return SubscriberClient.Reply.Ack;
                });
        }

        /// <summary>
        /// Parses the textual response from the client.
        /// </summary>
        /// <returns>The response text.</returns>
        /// <param name="jsonObject">The JSON payload sent back from the user.</param>
        private string GetResponseText(JObject jsonObject)
        {
            if (jsonObject["text"] != null)
            {
                return (string)jsonObject["text"];
            }
            else if (jsonObject["suggestionResponse"] != null)
            {
                JObject innerJson = (JObject)jsonObject["suggestionResponse"];

                return (string)innerJson.GetValue("postbackData");
            }

            return "";
        }

        /// <summary>
        /// Sends the initial greeting of "What is your favorite color?" to the user.
        /// </summary>
        public void SendGreeting()
        {
            string messageText = "What is your favorite color?";

            Console.WriteLine("Sending greeting to " + msisdn);

            rbmApiHelper.SendTextMessage(messageText, msisdn);
        }

        /// <summary>
        /// Sends the client a message stating that we like their color too.
        /// </summary>
        /// <param name="userResponse">The color choice by the user.</param>
        /// <param name="msisdn">The phone number, in E.164 format, to 
        /// send the message to.</param>
        public void HandleUserResponse(string userResponse, string msisdn)
        {
            if (userResponse.ToLower().Equals("stop"))
            {
                // Any real agent must support this command
                // TODO: Client typed stop, agent should no longer send messages to this msisdn
                Console.WriteLine(msisdn + " asked to stop agent messaging");
            }
            else
            {
                Console.WriteLine("Sending response to " + msisdn);

                // send typing indicator
                rbmApiHelper.SendIsTypingMessage(msisdn);

                string messageText = "I like " + userResponse + " too!";

                rbmApiHelper.SendTextMessage(messageText, msisdn);
            }
        }

        /// <summary>
        /// Sends a user an invite to test this agent.
        /// </summary>
        public void SendTesterInvite()
        {
            rbmApiHelper.RegsisterTester(msisdn);
        }

        /// <summary>
        /// Sends the client device a capability check.
        /// </summary>
        public void SendCapabilityCheck()
        {
            Capabilities capabilities = rbmApiHelper.GetCapability(msisdn);
            if (capabilities != null)
            {
                foreach (string feature in capabilities.Features)
                {
                    Console.WriteLine(feature);
                }
            }
        }

        static void Main(string[] args)
        {
            // make sure we have command line arguments
            if (args.Length != 3 && args.Length != 4)
            {
                Console.WriteLine("Usage:  " +
                        "<SERVICE_ACCOUNT.json> <GCP_PROJECT_ID> <PHONE E.164> <MODE>");

                Environment.Exit(0);
            }

            // get the credentials file and msisdn
            string credentialsFileLocation = args[0];
            string projectId = args[1];
            string msisdn = args[2];

            string mode = "chat";

            // check if a mode was passed in
            if (args.Length > 3)
            {
                mode = args[3];
            }

            // create a reference of the agent
            FirstAgent firstAgent = new FirstAgent(credentialsFileLocation,
                                                   projectId,
                                                   msisdn);

            // execute the method corresponding with the mode
            if (mode.Equals("chat"))
            {
                firstAgent.SendGreeting();

                while (true)
                {
                    Thread.Sleep(int.MaxValue);
                }
            }
            else if (mode.Equals("capcheck"))
            {
                Console.WriteLine("sending capability check");
                firstAgent.SendCapabilityCheck();
            }
            else if (mode.Equals("invite"))
            {
                Console.WriteLine("sending tester invite");
                firstAgent.SendTesterInvite();
            }
            else
            {
                Console.WriteLine("Invalid mode. Must be one of chat, capcheck, or invite.");
            }
        }
    }
}
