/**
 * Copyright 2016 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Utility functions for the Food Ordering Personal Assistant Test Agent.
 */
'use strict';

const Debug = require('debug');
const debug = Debug('actions-on-google:debug');
const error = Debug('actions-on-google:error');
const ActionsSdkAction = require('./actions-sdk-action');

const GoogleAuth = require('google-auth-library');
const request = require('request');
const promise = require('promise');
const constants = require('./constants.json');
// Path to private key file
const key = require('./sample-auth-key.json');

const RESPONSE_CODE_BAD_REQUEST = 400;
const ONE_DAY_IN_SECONDS = 1 * 24 * 60 * 60;
const FIFTEEN_MINUTES_IN_SECONDS = 15 * 60;
const ASSISTANT_AUTH_SCOPE = constants.assistantAuthScope;
const ASSISTANT_PUSH_MESSAGE_ENDPOINT = constants.assistantPushMessageEndpoint;
const NOT_FOUND_ITEM_ID = constants.notFoundItemId;

let auth = new GoogleAuth();
let authClient = new auth.JWT(
    key.client_email, null /* API_KEY */, key.private_key,
    [ASSISTANT_AUTH_SCOPE] /* Scopes */
    );

const FoodOrderingActions = class extends ActionsSdkAction {
  /**
   * Constructor for building a food ordering assistant.
   */
  constructor(options) {
    debug('FoodOrderingActions constructor');
    super(options);
  }
  /**
   * Validates checkout request for all required parameters. If no errors,
   * returns null. Sends a bad request error if basic things like cart are
   * missing. Sends an error response if partner related errors occur such as
   * item not found, delivery issue etc.
   * @param {Object} input Checkout Request to be validated.
   * @return {Object} FoodErrorExtension object.
   */
  validateCheckoutRequest(input) {
    if (!input || !input.intent === 'actions.foodordering.intent.CHECKOUT') {
      return this.handleError_(
          'Intent is not actions.foodordering.intent.CHECKOUT.');
    }
    if (!input.arguments || !input.arguments.length > 0) {
      return this.handleError_('No arguments in inputs.');
    }
    const arg = input.arguments[0];
    if (!arg) {
      return this.handleError_(
          'No argument in inputs. Input:' + JSON.stringify(input));
    }
    if (!arg.extension) {
      return this.handleError_(
          'No extension in arguments. Arg:' + JSON.stringify(arg));
    }
    const cart = arg.extension;
    if (cart['@type'] != 'type.googleapis.com/google.actions.v2.orders.Cart') {
      return this.handleError_(
          'Not valid cart object in arguments. Cart:' + JSON.stringify(cart));
    }
    if (!cart.lineItems) {
      return null;
    }
    // Sample code to show errors.
    for (let item of cart.lineItems) {
      if (item.id == NOT_FOUND_ITEM_ID) {
        return [{
          'error': 'NOT_FOUND',
          'id': NOT_FOUND_ITEM_ID,
          'description': 'This item could not be found in our inventory.'
        }];
      }
    }
    return null;
  }

  /**
   * Validates submit order request for all required parameters. If no errors,
   * returns null. Otherwise returns BAD_REQUEST error.
   * @param {Object} input submit order Request to be validated.
   * @return {boolean} True if the request was valid.
   */
  validateSubmitOrderRequest(input) {
    if (!input || !input.intent === this.StandardIntents.TRANSACTION_DECISION) {
      this.handleError_('Intent is not actions.intent.TRANSACTION_DECISION.');
      return false;
    }
    if (!input.arguments || !input.arguments.length > 0) {
      this.handleError_('No arguments in inputs.');
      return false;
    }
    const arg = input.arguments[0];
    if (!arg.transactionDecisionValue) {
      this.handleError_(
          'No transactionDecisionValue in arguments. Arg:' +
          JSON.stringify(arg));
      return false;
    }
    return true;
  }

  /**
   * Builds an authenticated order update request and sends to assistant push
   * message service.
   * @param {string} actionOrderId Merchant supplied order id of the order to be
   *                 updated.
   * @param {string} orderStatus Order status for the update.
   * @param {string} isSandbox Sandbox bit for the order.
   * @return {Promise} A promise function which returns the response/error
   *                  received from Assistant server after posting an
   *                  orderUpdate request.
   */
  sendOrderUpdateRequestAsync(actionOrderId, orderStatus, isSandbox) {
    var that = this;
    return new promise(function(resolve, reject) {
      that.getAccessTokenAsync_().then(
          function(token) {
            var options =
                that.buildAsyncOrderUpdateRequest_(actionOrderId, orderStatus, isSandbox);
            options.headers.Authorization = 'Bearer ' + token;
            request(options, function(error, response, body) {
              var data = {request: options, response: null};
              if (error) {
                data.response = error;
                reject(data);
              } else if (body.error) {
                data.response = 'Code: ' + body.error.code +
                    ' Message: ' + body.error.message;
                reject(data);
              } else {
                data.response = 'Order update completed successfully!';
                resolve(data);
              }
            });
          },
          function(error) {
            reject(error.stack || error);
          });
    });
  }

  /**
   * Get the value corresponding to CART agument
   * @return {Object} Value of the argument extension in assistant request
   *                  payload.
   */
  getCart() {
    debug('getCart');
    const input = this.getTopInput_();
    if (input.intent === 'actions.foodordering.intent.CHECKOUT') {
      const arg = input.arguments[0];
      if (arg && arg.extension &&
          arg.extension['@type'] ==
              'type.googleapis.com/google.actions.v2.orders.Cart') {
        delete arg.extension['@type'];
        return arg.extension;
      }
    }
  }

  /**
   * Get the value corresponding to TRANSACTION_DECISION
   * @return {Object} Value of the argument called transactionDecisionValue
   *                  in request payload.
   */
  getTransactionDecisionValue() {
    debug('getTransactionDecisionValue');
    const input = this.getTopInput_();
    if (input.intent === this.StandardIntents.TRANSACTION_DECISION) {
      const arg = input.arguments[0];
      if (arg && arg.transactionDecisionValue) {
        return arg.transactionDecisionValue;
      }
    }
  }

  /**
   * Executes a direct action response
   * @param {Object} structuredResponse A structured response to Assistant
   *                 server
   * @return {Object} Final response to Assistant server.
   */
  doDirectActionResponse(structuredResponse) {
    if (!structuredResponse) {
      this.handleError_('Invalid structured response');
      return null;
    }

    const finalResponse = {'richResponse': {}};
    finalResponse.richResponse =
        this.buildRichResponse().addStructuredResponse(structuredResponse);
    return this.doResponse_(
        this.buildResponseHelper_(
            null /*conversationToken*/, false /*expectUserResponse*/,
            null /*expectedInput*/, finalResponse),
        200);
  }

  /**
   * Error handling response.
   * @param {Object} foodOrderErrors repeated FoodErrorExtension object.
   * @return {Object} Final response to Assistant server.
   */
  doErrorResponse(foodOrderErrors) {
    const structuredResponse = {
      'error': {
        '@type':
            'type.googleapis.com/google.actions.v2.orders.FoodErrorExtension',
        foodOrderErrors: foodOrderErrors
      }
    };
    return this.doDirectActionResponse(structuredResponse);
  }

  /**
   * Builds structured response for action.foodordering.intent.CHECKOUT
   * direct action intent.
   * @param {Object} order Order response in the form of ProposedOrder proto.
   * @param {Object} transactionConfig Contains payment parameters.
   * @return {Object} Checkout structured response.
   */
  buildCheckoutResponse(order, transactionConfig) {
    debug('buildCheckoutResponse');
    const orderOptions = {
      requestShippingAddress: transactionConfig.shippingAddressRequired
    };
    const paymentOptions = {};
    paymentOptions.googleProvidedOptions = {
      tokenizationParameters: {
        tokenizationType: transactionConfig.tokenizationType,
        parameters: transactionConfig.tokenizationParameters
      },
      supportedCardNetworks: transactionConfig.supportedCardNetworks,
      prepaidCardDisallowed: transactionConfig.prepaidCardDisallowed
    };

    const structuredResponse = {
      checkoutResponse: {
        proposedOrder: order,
        orderOptions: orderOptions,
        paymentOptions: paymentOptions
      }
    };

    return structuredResponse;
  }

  /**
   * Builds structured response for action.intent.TRANSACTION_DECISION
   * direct action intent.
   * @return {Object} OrderUpdate structured response.
   */
  buildTransactionResponse() {
    debug('buildTransactionResponse');

    const orderState = {
      state: 'CREATED',
      label: 'Order is created with partner.'
    };
    const structuredResponse = {
      orderUpdate: {
        actionOrderId: this.generateUniqueId_(),
        orderState: orderState,
        updateTime: (new Date()).toISOString(),
        orderManagementActions: this.buildOrderActions_()
      }
    };

    return structuredResponse;
  }

  /**
   * Build a proposed order using provider created fields such as cart,
   * deliveryFees, tax. Assigns a random order id to every order.
   * @param {Object} cart Cart object retrieved from the checkout request.
   * @param {Object} totalPrice Money object containing the total price.
   * @param {Object} deliveryFees LineItem object denoting delivery fees.
   * @param {Object} tax LineItem object denoting tax.
   * @return {Object} ProposedOrder object.
   */
  buildProposedOrder(cart, totalPrice, deliveryFees, tax) {
    const availableFulfillmentInfo = this.buildAvailableFulfillmentInfo_(cart);
    const order = this.buildOrder()
                      .setCart(cart)
                      .addOtherItems([deliveryFees, tax])
                      .setTotalPrice(
                          this.Transactions.PriceType.ESTIMATE, 'USD',
                          totalPrice.units.toString(), totalPrice.nanos);
    order.extension = {
      '@type':
          'type.googleapis.com/google.actions.v2.orders.FoodOrderExtension',
      availableFulfillmentOptions: [availableFulfillmentInfo]
    };
    return order;
  }

  // ---------------------------------------------------------------------------
  //                   Private Helpers
  // ---------------------------------------------------------------------------
  /**
   * Calls the google auth library to get an auth token using a private key
   * file.
   * @return {string} Access token
   * @private
   */
  getAccessTokenAsync_() {
    return new promise(function(resolve, reject) {
      authClient.authorize(function(err, tokens) {
        if (err) {
          reject(err);
        } else {
          resolve(tokens.access_token);
        }
      });
    });
  }

  /**
   * Builds an order update request to be sent to assistant. The authorization
   * header is not set here.
   * @param {string} actionOrderId The merchant supplied order id of the order
   *                 to be updated.
   * @param {string} orderStatus Order status for the update.
   * @param {string} isSandbox Sandbox bit associated with the order.
   * @return {Object} An object of type http post request with fields such as
   *                  headers, payload etc. set appropriately.
   * @private
   */
  buildAsyncOrderUpdateRequest_(actionOrderId, orderStatus, isSandbox) {
    var headers = {
      'Content-Type': 'application/json',
    };
    var orderState = {state: orderStatus, label: 'Order is updated.'};
    var updateTime = (new Date()).toISOString();
    var receipt = {userVisibleOrderId: this.generateUniqueId_()};
    var orderUpdate = {
      orderUpdate: {
        actionOrderId: actionOrderId,
        orderState: orderState,
        updateTime: updateTime,
        orderManagementActions: this.buildOrderActions_(),
        receipt: receipt,
        infoExtension: {
          '@type':
              'type.googleapis.com/google.actions.v2.orders.FoodOrderUpdateExtension',
          'estimatedFulfillmentTimeIso8601': 'PT20M'
        }
      }
    };
    var postData = {
      customPushMessage: orderUpdate,
      isInSandbox: isSandbox == 'false' ? false : true
    };
    var options = {
      url: ASSISTANT_PUSH_MESSAGE_ENDPOINT,
      method: 'POST',
      headers: headers,
      body: postData,
      json: true
    };
    return options;
  }

  /**
   * Parses checkout request for delivery/pickup options and returns these
   * options accordingly, if available.
   * @param {Object} cart Cart object containing items selected by user.
   * @return {Object} Returns an object having these fields:
   *                  delivery/pickup: User's choice regarding delivery/pickup.
   *                  expiresAt: Expiry time of the order.
   *                  price: The total price of all items in the cart.
   * @private
   */
  buildAvailableFulfillmentInfo_(cart) {
    if (!(cart.extension &&
          cart.extension['@type'] ==
              'type.googleapis.com/google.actions.v2.orders.FoodCartExtension' &&
          cart.extension.fulfillmentPreference &&
          cart.extension.fulfillmentPreference.fulfillmentInfo)) {
      return {};
    }
    let availableFulfillmentInfo = {};
    const requestedFulfillmentInfo =
        cart.extension.fulfillmentPreference.fulfillmentInfo;
    let delivery = requestedFulfillmentInfo.delivery;
    let pickup = requestedFulfillmentInfo.pickup;

    if (delivery) {
      let requestedDeliveryTime = delivery.deliveryTimeIso8601;
      // Check if requested time can be served., and suggest delivery window.
      availableFulfillmentInfo = {
        fulfillmentInfo: {
          delivery: {deliveryTimeIso8601: requestedDeliveryTime},
        },
        expiresAt: (new Date()).toISOString(),
        price: {currencyCode: 'USD', units: 4, nanos: 990000000}
      };
    } else if (pickup) {
      let requestedPickupTime = pickup.pickupTimeIso8601;
      // Check if requested pickup time can be served and suggest pickup time.
      availableFulfillmentInfo = {
        fulfillmentInfo: {
          pickup: {pickupTimeIso8601: requestedPickupTime},
        },
        expiresAt: (new Date()).toISOString()
      };
    }
    return availableFulfillmentInfo;
  }

  /**
   * Generates a string unique id..
   * @return {string} Unique string id.
   * @private
   */
  generateUniqueId_() {
    var currentTimestamp = Date.now();
    // Generates a random number between 1 and 1000.
    var randomid = Math.floor(Math.random() * 999 + 1);
    return currentTimestamp + ':' + randomid;
  }

  /**
   * Returns true if request is in sandbox mode.
   *
   * @return {boolean} True if request is in sandbox mode, otherwise false.
   */
  isSandbox() {
    debug('isSandbox');
    return (!this.body_.isInSandbox) ? false : this.body_.isInSandbox;
  }

  /**
   * Returns a list of order management actions.
   * @return {Object} Array of order actions.
   *
   * @private
   */
  buildOrderActions_() {
    var orderActions = [
      {
        type: 'CALL',
        button:
            {title: 'Call Us', openUrlAction: {url: 'tel:+1-111-111-1111'}}
      },
      {
        type: 'EMAIL',
        button: {
          title: 'Email Us',
          openUrlAction: {url: 'mailto:johndoe@gmail.com'}
        }
      },
      {
        type: 'CUSTOMER_SERVICE',
        button: {
          title: 'Customer Service',
          openUrlAction: {url: 'http://www.google.com'}
        }
      }
    ];
    return orderActions;
  }
};

module.exports = FoodOrderingActions;

