var testParams = require('./test_params.json');

/**
 * Checks given response object for basic proto format errors.
 * @param {Object} request The checkout request received by partner.
 * @param {Object} res The response object returned by partner's checkout
 *                     endpoint on being sent a checkout request.
 */
module.exports.testInitialStructuredResponse = function(request, res) {
  res.should.have.status(200);
  res.should.have.property('body');
  var body = res.body;
  body.should.has.property('finalResponse')
      .that.has.property('richResponse')
      .that.has.property('items');
  var items = res.body.finalResponse.richResponse.items;
  items.should.be.a('array').that.has.length.of(1);
  var item = items[0];
  item.should.has.property('structuredResponse');
};

/**
 * Checks given response object for cart and compares it to the one in request.
 * @param {Object} request The checkout request received by partner.
 * @param {Object} res The response object returned by partner's checkout
 *                     endpoint on being sent a checkout request.
 */
module.exports.testCartCheckoutResponse = function(request, res) {
  var expectedCartValue = request.inputs[0].arguments[0].extension;
  delete expectedCartValue['@type'];
  var structuredResponse =
      res.body.finalResponse.richResponse.items[0].structuredResponse;
  structuredResponse.should.has.property('checkoutResponse');
  var checkoutResponse = structuredResponse.checkoutResponse;
  checkoutResponse.should.has.property('proposedOrder')
      .that.has.property('cart')
      .that.deep.equals(expectedCartValue);
};

/**
 * Checks given response object for paymentOptions.
 * @param {Object} request The checkout request received by partner.
 * @param {Object} res The response object returned by partner's checkout
 *                     endpoint on being sent a checkout request.
 */
module.exports.testPaymentOptionsCheckoutResponse = function(request, res) {
  var checkoutResponse = res.body.finalResponse.richResponse.items[0]
                             .structuredResponse.checkoutResponse;
  checkoutResponse.should.has.property('paymentOptions')
      .that.has.property('googleProvidedOptions')
      .that.has.property('tokenizationParameters');
  checkoutResponse.paymentOptions.googleProvidedOptions.should.has.property(
      'supportedCardNetworks');
  var tokenParams = checkoutResponse.paymentOptions.googleProvidedOptions
                        .tokenizationParameters;
  tokenParams.should.has.property('tokenizationType');
  tokenParams.should.has.property('parameters');
};

/**
 * Checks given response object for total_price.
 * @param {Object} request The checkout request received by partner.
 * @param {Object} res The response object returned by partner's checkout
 *                     endpoint on being sent a checkout request.
 */
module.exports.testTotalPriceCheckoutResponse = function(request, res) {
  var proposedOrder = res.body.finalResponse.richResponse.items[0]
                          .structuredResponse.checkoutResponse.proposedOrder;
  proposedOrder.should.has.property('totalPrice').that.has.property('type');
  proposedOrder.should.has.property('totalPrice').that.has.property('amount');
  var price = proposedOrder.totalPrice.amount;
  price.should.has.property('currencyCode');
  price.should.has.property('units');
  price.should.has.property('nanos');
};

/**
 * Checks given response object for food order extension.
 * @param {Object} request The checkout request received by partner.
 * @param {Object} res The response object returned by partner's checkout
 *                     endpoint on being sent a checkout request.
 */
module.exports.testFoodOrderExtensionCheckoutResponse = function(request, res) {
  var proposedOrder = res.body.finalResponse.richResponse.items[0]
                          .structuredResponse.checkoutResponse.proposedOrder;
  proposedOrder.should.has.property('extension')
      .that.has.property('availableFulfillmentOptions')
      .that.has.length.above(0);
};

/**
 * Checks given response object for other_items.
 * @param {Object} request The checkout request received by partner.
 * @param {Object} res The response object returned by partner's checkout
 *                     endpoint on being sent a checkout request.
 */
module.exports.testProposedOrderOtherItemsCheckoutResponse = function(
    request, res) {
  var proposedOrder = res.body.finalResponse.richResponse.items[0]
                          .structuredResponse.checkoutResponse.proposedOrder;
  var otherItems = proposedOrder.otherItems;
  if (otherItems) {
    otherItems.forEach(function(item) {
      item.should.has.property('type');
      item.should.has.property('price');
    });
  }
};

/**
 * Tests whether actionOrderId is present in submit response. It can be a
 * different value than the googleOrderId in submit request.
 * order response
 * @param {Object} request The submit order request received by partner.
 * @param {Object} res The response object returned by partner's submit order
 *                     endpoint on being sent a checkout request.
 */
module.exports.testActionOrderIdSubmitOrderResponse = function(request, res) {
  var orderUpdate = res.body.finalResponse.richResponse.items[0]
                        .structuredResponse.orderUpdate;
  orderUpdate.should.has.property('actionOrderId');
};

/**
 * Builds the basic checkout request with empty cart_value.
 * @return {Object} Checkout request object.
 */
module.exports.buildInitialCheckoutRequest = function() {
  var request = {};
  request.user = {};
  request.device = {};
  request.conversation = {};
  request.inputs = [];
  request.inputs[0] = {
    intent: 'actions.foodordering.intent.CHECKOUT',
    arguments: [{}]
  };
  return request;
};

/**
 * Builds the basic submit order request with empty order.
 * @return {Object} Submit order request object.
 */
module.exports.buildInitialSubmitOrderRequest = function() {
  var request = {};
  request.user = {};
  request.device = {};
  request.conversation = {};
  request.inputs = [];
  request.inputs[0] = {
    intent: 'actions.intent.TRANSACTION_DECISION',
    arguments: [{
      'transactionDecisionValue':
          {'order': {'finalOrder': {'cart': {}}, 'googleOrderId': '1234'}}
    }]
  };
  return request;
};
/**
 * Builds the basic checkout request with a cart having delivery, merchant etc.
 * specified but no items.
 * @return {Object} Checkout request object.
 */
module.exports.buildCartWithNoItems = function() {
  // add postal address
  var cart = {
    '@type': 'type.googleapis.com/google.actions.v2.orders.Cart',
    'extension': {
      '@type': 'type.googleapis.com/google.actions.v2.orders.FoodCartExtension',
      'fulfillmentPreference': {
        'fulfillmentInfo': {
          'delivery': {'deliveryTimeIso8601': testParams.deliveryTimeIso8601}
        }
      },
      'location': {

        'formattedAddress': testParams.deliveryLocationFormattedAddress
      }
    },
    'notes': 'Some order notes...',
    'merchant': {'name': testParams.merchantName, 'id': testParams.merchantId},
    'lineItems': []
  };
  return cart;
};

