/**
 * Copyright 2016 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * This is the class that handles the conversation API directly from Assistant,
 * providing implementation for all the methods available in the API.
 */

'use strict';

const Debug = require('debug');
const debug = Debug('actions-on-google:debug');
const error = Debug('actions-on-google:error');
const transformToSnakeCase = require('./utils/transform').transformToSnakeCase;
const action = require('./action');
const Action = action.Action;
const State = action.State;

// Constants
const CONVERSATION_API_AGENT_VERSION_HEADER = 'Agent-Version-Label';
const RESPONSE_CODE_OK = 200;
const INPUTS_MAX = 3;
const CONVERSATION_API_SIGNATURE_HEADER = 'authorization';

// Configure logging for hosting platforms that only support console.log and console.error
debug.log = console.log.bind(console);
error.log = console.error.bind(console);

// ---------------------------------------------------------------------------
//                   Actions SDK support
// ---------------------------------------------------------------------------

/**
 * Constructor for ActionsSdkAction object. To be used in the Actions SDK
 * HTTP endpoint logic.
 *
 * @example
 * const ActionsSdkAction = require('actions-on-google').ActionsSdkAction;
 * const action = new ActionsSdkAction({request: request, response: response,
 *   sessionStarted:sessionStarted});
 *
 * @param {Object} options JSON configuration.
 * @param {Object} options.request Express HTTP request object.
 * @param {Object} options.response Express HTTP response object.
 * @param {Function=} options.sessionStarted Function callback when session starts.
 * @actionssdk
 */
const ActionsSdkAction = class extends Action {
  constructor (options) {
    debug('ActionsSdkAction constructor');
    super(options);

    if (this.body_ &&
        this.body_.conversation &&
        this.body_.conversation.type &&
        this.body_.conversation.type === this.ConversationStages.NEW &&
        this.sessionStarted_ && typeof this.sessionStarted_ === 'function') {
      this.sessionStarted_();
    } else if (this.sessionStarted_ && typeof this.sessionStarted_ !== 'function') {
      this.handleError_('options.sessionStarted must be a Function');
    }
  }

  /*
   * Gets the request Conversation API version.
   *
   * @example
   * const action = new ActionsSdkAction({request: request, response: response});
   * const apiVersion = action.getApiVersion();
   *
   * @return {string} Version value or null if no value.
   * @actionssdk
   */
  getApiVersion () {
    debug('getApiVersion');
    return this.apiVersion_ || this.actionsApiVersion_;
  }

  /**
   * Gets the user's raw input query.
   *
   * @example
   * const action = new ActionsSdkAction({request: request, response: response});
   * action.tell('You said ' + action.getRawInput());
   *
   * @return {string} User's raw query or null if no value.
   * @actionssdk
   */
  getRawInput () {
    debug('getRawInput');
    const input = this.getTopInput_();
    if (!input) {
      this.handleError_('Failed to get top Input.');
      return null;
    }
    if (!input.rawInputs || input.rawInputs.length === 0) {
      this.handleError_('Missing user raw input');
      return null;
    }
    const rawInput = input.rawInputs[0];
    if (!rawInput.query) {
      this.handleError_('Missing query for user raw input');
      return null;
    }
    return rawInput.query;
  }

  /**
   * Gets previous JSON dialog state that the action sent to Assistant.
   * Alternatively, use the action.data field to store JSON values between requests.
   *
   * @example
   * const action = new ActionsSdkAction({request: request, response: response});
   * const dialogState = action.getDialogState();
   *
   * @return {Object} JSON object provided to the Assistant in the previous
   *     user turn or {} if no value.
   * @actionssdk
   */
  getDialogState () {
    debug('getDialogState');
    if (this.body_.conversation && this.body_.conversation.conversationToken) {
      return JSON.parse(this.body_.conversation.conversationToken);
    }
    return {};
  }

  /**
   * Gets the {@link User} object.
   * The user object contains information about the user, including
   * a string identifier and personal information (requires requesting permissions,
   * see {@link Action#askForPermissions|askForPermissions}).
   *
   * @example
   * const action = new ActionsSdkAction({request: request, response: response});
   * const userId = action.getUser().userId;
   *
   * @return {User} Null if no value.
   * @actionssdk
   */
  getUser () {
    debug('getUser');
    if (!this.body_.user) {
      this.handleError_('No user object');
      return null;
    }
    // User object includes original API properties
    const user = {
      userId: this.body_.user.userId,
      user_id: this.body_.user.userId,
      userName: this.body_.user.profile ? {
        displayName: this.body_.user.profile.displayName,
        givenName: this.body_.user.profile.givenName,
        familyName: this.body_.user.profile.familyName
      } : null,
      profile: this.body_.user.profile,
      accessToken: this.body_.user.accessToken,
      access_token: this.body_.user.accessToken
    };
    return user;
  }

  /**
   * Returns true if user device has a given capability.
   *
   * @param {string} capability Must be one of ActionsSdkAction.SurfaceCapabilities.
   * @return {boolean} True if user device has given capability.
   * @actionssdk
   */
  hasSurfaceCapability (requestedCapability) {
    debug('hasSurfaceCapability');
    if (!(this.body_.surface &&
      this.body_.surface.capabilities)) {
      this.handleError_('No surface capabilities in incoming request');
      return null;
    }
    for (let capability of this.body_.surface.capabilities) {
      if (capability.name === requestedCapability) {
        return true;
      }
    }
    return false;
  }

  /**
   * Gets surface capabilities of user device.
   *
   * @return {Array<string>} Supported surface capabilities, as defined in
   *     ActionsSdkAction.SurfaceCapabilities.
   * @actionssdk
   */
  getSurfaceCapabilities () {
    debug('getSurfaceCapabilities');
    if (!(this.body_.surface &&
      this.body_.surface.capabilities)) {
      this.handleError_('No surface capabilities in incoming request');
      return null;
    }
    const capabilities = [];
    for (let capability of this.body_.surface.capabilities) {
      capabilities.push(capability.name);
    }
    return capabilities;
  }

  /**
   * Gets type of input given in this request.
   *
   * @return {number} One of ActionsSdkAction.InputTypes. Null if no input type given.
   * @actionssdk
   */
  getInputType () {
    debug('getInputType');
    if (this.body_ && this.body_.inputs) {
      for (let input of this.body_.inputs) {
        if (input.raw_inputs) {
          for (let rawInput of input.rawInputs) {
            if (rawInput.inputType) {
              return rawInput.inputType;
            }
          }
        }
      }
    } else {
      this.handleError_('No input type in incoming request');
      return null;
    }
  }

  /**
   * If granted permission to device's location in previous intent, returns device's
   * location (see {@link Action#askForPermissions|askForPermissoins}).
   * If device info is unavailable, returns null.
   *
   * @example
   * const action = new ActionsSdkAction({request: req, response: res});
   * action.askForPermission("To get you a ride",
   *   action.SupportedPermissions.DEVICE_PRECISE_LOCATION);
   * // ...
   * // In response handler for subsequent intent:
   * if (action.isPermissionGranted()) {
   *   sendCarTo(action.getDeviceLocation().coordinates);
   * }
   *
   * @return {DeviceLocation} Null if location permission is not granted.
   * @actionssdk
   */
  getDeviceLocation () {
    debug('getDeviceLocation');
    if (!this.body_.device || !this.body_.device.location) {
      return null;
    }
    const deviceLocation = {
      coordinates: this.body_.device.location.coordinates,
      address: this.body_.device.location.formattedAddress,
      zipCode: this.body_.device.location.zipCode,
      city: this.body_.device.location.city
    };
    return deviceLocation;
  }

  /**
   * Returns true if the request follows a previous request asking for
   * permission from the user and the user granted the permission(s). Otherwise,
   * false. Use with {@link Action#askForPermissions|askForPermissions}.
   *
   * @example
   * const action = new ActionsSdkAction({request: request, response: response});
   * action.askForPermissions("To get you a ride", [
   *   action.SupportedPermissions.NAME,
   *   action.SupportedPermissions.DEVICE_PRECISE_LOCATION
   * ]);
   * // ...
   * // In response handler for subsequent intent:
   * if (action.isPermissionGranted()) {
   *  // Use the requested permission(s) to get the user a ride
   * }
   *
   * @return {boolean} true if permissions granted.
   * @actionssdk
   */
  isPermissionGranted () {
    debug('isPermissionGranted');
    return this.getArgument(this.BuiltInArgNames.PERMISSION_GRANTED) === 'true';
  }

  /**
   * Gets the "versionLabel" specified inside the Action Package.
   * Used by actions to do version control.
   *
   * @example
   * const action = new ActionsSdkAction({request: request, response: response});
   * const actionVersionLabel = action.getActionVersionLabel();
   *
   * @return {string} The specified version label or null if unspecified.
   * @actionssdk
   */
  getActionVersionLabel () {
    debug('getActionVersionLabel');
    const versionLabel = this.request_.get(CONVERSATION_API_AGENT_VERSION_HEADER);
    if (versionLabel) {
      return versionLabel;
    } else {
      return null;
    }
  }

  /**
   * Gets the unique conversation ID. It's a new ID for the initial query,
   * and stays the same until the end of the conversation.
   *
   * @example
   * const action = new ActionsSdkAction({request: request, response: response});
   * const conversationId = action.getConversationId();
   *
   * @return {string} Conversation ID or null if no value.
   * @actionssdk
   */
  getConversationId () {
    debug('getConversationId');
    if (!this.body_.conversation || !this.body_.conversation.conversationId) {
      this.handleError_('No conversation ID');
      return null;
    }
    return this.body_.conversation.conversationId;
  }

  /**
   * Get the current intent. Alternatively, using a handler Map with
   * {@link Action#handleRequest|handleRequest}, the client library will
   * automatically handle the incoming intents.
   *
   * @example
   * const action = new ActionsSdkAction({request: request, response: response});
   *
   * function responseHandler (action) {
   *   const intent = action.getIntent();
   *   switch (intent) {
   *     case action.StandardIntents.MAIN:
   *       const inputPrompt = action.buildInputPrompt(false, 'Welcome to action snippets! Say anything.');
   *       action.ask(inputPrompt);
   *       break;
   *
   *     case action.StandardIntents.TEXT:
   *       action.tell('You said ' + action.getRawInput());
   *       break;
   *   }
   * }
   *
   * action.handleRequest(responseHandler);
   *
   * @return {string} Intent id or null if no value.
   * @actionssdk
   */
  getIntent () {
    debug('getIntent');
    const input = this.getTopInput_();
    if (!input) {
      this.handleError_('Missing intent from request body');
      return null;
    }
    return input.intent;
  }

  /**
   * Get the argument value by name from the current intent. If the argument
   * is not a text argument, the entire argument object is returned.
   *
   * @param {string} argName Name of the argument.
   * @return {string} Argument value matching argName
   *     or null if no matching argument.
   * @actionssdk
   */
  getArgument (argName) {
    debug('getArgument: argName=%s', argName);
    if (!argName) {
      this.handleError_('Invalid argument name');
      return null;
    }
    const argument = this.getArgument_(argName);
    if (!argument) {
      debug('Failed to get argument value: %s', argName);
      return null;
    } else if (argument.textValue) {
      return argument.textValue;
    } else {
      if (!this.isProto3_()) {
        return transformToSnakeCase(argument);
      } else {
        return argument;
      }
    }
  }

  /**
   * Asks Assistant to collect user's input; all user's queries need to be sent to
   * the action.
   *
   * @example
   * const action = new ActionsSdkAction({request: request, response: response});
   *
   * function mainIntent (action) {
   *   const inputPrompt = action.buildInputPrompt(true, '<speak>Hi! <break time="1"/> ' +
   *         'I can read out an ordinal like ' +
   *         '<say-as interpret-as="ordinal">123</say-as>. Say a number.</speak>',
   *         ['I didn\'t hear a number', 'If you\'re still there, what\'s the number?', 'What is the number?']);
   *   action.ask(inputPrompt);
   * }
   *
   * function rawInput (action) {
   *   if (action.getRawInput() === 'bye') {
   *     action.tell('Goodbye!');
   *   } else {
   *     const inputPrompt = action.buildInputPrompt(true, '<speak>You said, <say-as interpret-as="ordinal">' +
   *       action.getRawInput() + '</say-as></speak>',
   *         ['I didn\'t hear a number', 'If you\'re still there, what\'s the number?', 'What is the number?']);
   *     action.ask(inputPrompt);
   *   }
   * }
   *
   * const actionMap = new Map();
   * actionMap.set(action.StandardIntents.MAIN, mainIntent);
   * actionMap.set(action.StandardIntents.TEXT, rawInput);
   *
   * action.handleRequest(actionMap);
   *
   * @param {Object} inputPrompt Holding rich, initial and no-input prompts.
   * @param {string} inputPrompt.speech If using SimpleResponse, speech to be spoken
   *     to user.
   * @param {string=} inputPrompt.displayText If using SimpleResponse, text to be shown
   *     to user.
   * @param {Object=} dialogState JSON object the action uses to hold dialog state that
   *     will be circulated back by Assistant.
   * @return The response that is sent to Assistant to ask user to provide input.
   * @actionssdk
   */
  ask (inputPrompt, dialogState) {
    debug('ask: inputPrompt=%s, dialogState=%s',
       JSON.stringify(inputPrompt), JSON.stringify(dialogState));
    const expectedIntent = this.buildExpectedIntent_(this.StandardIntents.TEXT, []);
    return this.buildAskHelper_(inputPrompt, [expectedIntent], dialogState);
  }

  /**
   * Asks Assistant to collect user's input with a list.
   *
   * @param {Object} inputPrompt Holding rich, initial and no-input prompts.
   * @param {string} inputPrompt.speech If using SimpleResponse, speech to be spoken
   *     to user.
   * @param {string=} inputPrompt.displayText If using SimpleResponse, text to be shown
   *     to user.
   * @param {Object} list List built with ActionsSdk.buildList.
   * @param {Object=} dialogState JSON object the action uses to hold dialog state that
   *     will be circulated back by Assistant.
   * @return The response that is sent to Assistant to ask user to provide input.
   * @actionssdk
   */
  askWithList (inputPrompt, list, dialogState) {
    debug('ask: inputPrompt=%s, dialogState=%s',
      JSON.stringify(inputPrompt), JSON.stringify(dialogState));
    const expectedIntent = this.buildExpectedIntent_(this.StandardIntents.OPTION, []);
    expectedIntent.inputValueSpec = {
      optionValueSpec: {
        listSelect: list
      }
    };
    return this.buildAskHelper_(inputPrompt, [expectedIntent], dialogState);
  }

  /**
   * Asks Assistant to collect user's input with a carousel.
   *
   * @param {Object} inputPrompt Holding rich, initial and no-input prompts.
   * @param {string} inputPrompt.speech If using SimpleResponse, speech to be spoken
   *     to user.
   * @param {string=} inputPrompt.displayText If using SimpleResponse, text to be shown
   *     to user.
   * @param {Object} carousel Carousel built with ActionsSdk.buildCarousel.
   * @param {Object=} dialogState JSON object the action uses to hold dialog state that
   *     will be circulated back by Assistant.
   * @return The response that is sent to Assistant to ask user to provide input.
   * @actionssdk
   */
  askWithCarousel (inputPrompt, carousel, dialogState) {
    debug('ask: inputPrompt=%s, dialogState=%s',
      JSON.stringify(inputPrompt), JSON.stringify(dialogState));
    const expectedIntent = this.buildExpectedIntent_(this.StandardIntents.OPTION, []);
    expectedIntent.inputValueSpec = {
      optionValueSpec: {
        carouselSelect: carousel
      }
    };
    return this.buildAskHelper_(inputPrompt, [expectedIntent], dialogState);
  }

  /**
   * Tells Assistant to render the speech response and close the mic.
   *
   * @example
   * const action = new ActionsSdkAction({request: request, response: response});
   *
   * function mainIntent (action) {
   *   const inputPrompt = action.buildInputPrompt(true, '<speak>Hi! <break time="1"/> ' +
   *         'I can read out an ordinal like ' +
   *         '<say-as interpret-as="ordinal">123</say-as>. Say a number.</speak>',
   *         ['I didn\'t hear a number', 'If you\'re still there, what\'s the number?', 'What is the number?']);
   *   action.ask(inputPrompt);
   * }
   *
   * function rawInput (action) {
   *   if (action.getRawInput() === 'bye') {
   *     action.tell('Goodbye!');
   *   } else {
   *     const inputPrompt = action.buildInputPrompt(true, '<speak>You said, <say-as interpret-as="ordinal">' +
   *       action.getRawInput() + '</say-as></speak>',
   *         ['I didn\'t hear a number', 'If you\'re still there, what\'s the number?', 'What is the number?']);
   *     action.ask(inputPrompt);
   *   }
   * }
   *
   * const actionMap = new Map();
   * actionMap.set(action.StandardIntents.MAIN, mainIntent);
   * actionMap.set(action.StandardIntents.TEXT, rawInput);
   *
   * action.handleRequest(actionMap);
   *
   * @param {string} textToSpeech Final rich/spoken response. Spoken response can be SSML.
   * @param {string} textToSpeech.speech If using SimpleResponse, speech to be spoken
   *     to user.
   * @param {string=} textToSpeech.displayText If using SimpleResponse, text to be shown
   *     to user.
   * @return The HTTP response that is sent back to Assistant.
   * @actionssdk
   */
  tell (textToSpeech) {
    debug('tell: textToSpeech=%s', textToSpeech);
    if (!textToSpeech) {
      this.handleError_('Invalid speech response');
      return null;
    }
    const finalResponse = {};
    if (typeof textToSpeech === 'string') {
      if (this.isSsml_(textToSpeech)) {
        finalResponse.speechResponse = {
          ssml: textToSpeech
        };
      } else {
        finalResponse.speechResponse = {
          textToSpeech: textToSpeech
        };
      }
    } else {
      if (textToSpeech.items) { // Check for RichResponse
        finalResponse.richResponse = textToSpeech;
      } else if (textToSpeech.speech) { // Check for SimpleResponse
        finalResponse.richResponse = this.buildRichResponse()
          .addSimpleResponse(textToSpeech);
      } else {
        this.handleError_('Invalid speech response. Must be string, ' +
          'RichResponse or SimpleResponse.');
        return null;
      }
    }
    const response = this.buildResponseHelper_(null, false, null, finalResponse);
    return this.doResponse_(response, RESPONSE_CODE_OK);
  }

  /**
   * Builds the {@link https://developers.google.com/actions/reference/conversation#InputPrompt|InputPrompt object}
   * from initial prompt and no-input prompts.
   *
   * The Action needs one initial prompt to start the conversation. If there is no user response,
   * the Action re-opens the mic and renders the no-input prompts three times
   * (one for each no-input prompt that was configured) to help the user
   * provide the right response.
   *
   * Note: we highly recommend action to provide all the prompts required here in order to ensure a
   * good user experience.
   *
   * @example
   * const inputPrompt = action.buildInputPrompt(false, 'Welcome to action snippets! Say a number.',
   *     ['Say any number', 'Pick a number', 'What is the number?']);
   * action.ask(inputPrompt);
   *
   * @param {boolean} isSsml Indicates whether the text to speech is SSML or not.
   * @param {string} initialPrompt The initial prompt the Action asks the user.
   * @param {Array<string>=} noInputs Array of re-prompts when the user does not respond (max 3).
   * @return {Object} An {@link https://developers.google.com/actions/reference/conversation#InputPrompt|InputPrompt object}.
   * @actionssdk
   */
  buildInputPrompt (isSsml, initialPrompt, noInputs) {
    debug('buildInputPrompt: isSsml=%s, initialPrompt=%s, noInputs=%s',
      isSsml, initialPrompt, noInputs);
    const initials = [];

    if (noInputs) {
      if (noInputs.length > INPUTS_MAX) {
        this.handleError_('Invalid number of no inputs');
        return null;
      }
    } else {
      noInputs = [];
    }

    this.maybeAddItemToArray_(initialPrompt, initials);
    if (isSsml) {
      return {
        initialPrompts: this.buildPromptsFromSsmlHelper_(initials),
        noInputPrompts: this.buildPromptsFromSsmlHelper_(noInputs)
      };
    } else {
      return {
        initialPrompts: this.buildPromptsFromPlainTextHelper_(initials),
        noInputPrompts: this.buildPromptsFromPlainTextHelper_(noInputs)
      };
    }
  }

// ---------------------------------------------------------------------------
//                   Private Helpers
// ---------------------------------------------------------------------------

  /**
   * Get the top most Input object.
   *
   * @return {object} Input object.
   * @private
   * @actionssdk
   */
  getTopInput_ () {
    debug('getTopInput_');
    if (!this.body_.inputs || this.body_.inputs.length === 0) {
      this.handleError_('Missing inputs from request body');
      return null;
    }
    return this.body_.inputs[0];
  }

  /**
   * Builds the response to send back to Assistant.
   *
   * @param {string} conversationToken The dialog state.
   * @param {boolean} expectUserResponse The expected user response.
   * @param {Object} expectedInput The expected response.
   * @param {boolean} finalResponse The final response.
   * @return {Object} Final response returned to server.
   * @private
   * @actionssdk
   */
  buildResponseHelper_ (conversationToken, expectUserResponse, expectedInput, finalResponse) {
    debug('buildResponseHelper_: conversationToken=%s, expectUserResponse=%s, ' +
      'expectedInput=%s, finalResponse=%s',
      conversationToken, expectUserResponse, JSON.stringify(expectedInput),
      JSON.stringify(finalResponse));
    const response = {};
    if (conversationToken) {
      response.conversationToken = conversationToken;
    }
    response.expectUserResponse = expectUserResponse;
    if (expectedInput) {
      response.expectedInputs = expectedInput;
    }
    if (!expectUserResponse && finalResponse) {
      response.finalResponse = finalResponse;
    }
    return response;
  }

  /**
   * Helper to add item to an array.
   *
   * @private
   * @actionssdk
   */
  maybeAddItemToArray_ (item, array) {
    debug('maybeAddItemToArray_: item=%s, array=%s', item, array);
    if (!array) {
      this.handleError_('Invalid array');
      return;
    }
    if (!item) {
      // ignore add
      return;
    }
    array.push(item);
  }

  /**
   * Get the argument by name from the current action.
   *
   * @param {string} argName Name of the argument.
   * @return {Object} Argument value matching argName
         or null if no matching argument.
   * @private
   * @actionssdk
   */
  getArgument_ (argName) {
    debug('getArgument_: argName=%s', argName);
    if (!argName) {
      this.handleError_('Invalid argument name');
      return null;
    }
    const input = this.getTopInput_();
    if (!input) {
      this.handleError_('Missing action');
      return null;
    }
    if (!arguments) {
      debug('No arguments included in request');
      return null;
    }
    for (let i = 0; i < input.arguments.length; i++) {
      if (input.arguments[i].name === argName) {
        return input.arguments[i];
      }
    }
    debug('Failed to find argument: %s', argName);
    return null;
  }

  /**
   * Extract session data from the incoming JSON request.
   *
   * @private
   * @actionssdk
   */
  extractData_ () {
    debug('extractData_');
    if (this.body_.conversation &&
      this.body_.conversation.conversationToken) {
      const json = JSON.parse(this.body_.conversation.conversationToken);
      this.data = json.data;
      this.state = json.state;
    } else {
      this.data = {};
    }
  }

  /**
   * Uses a PermissionsValueSpec object to construct and send a
   * permissions request to user.
   *
   * @param {Object} permissionsSpec PermissionsValueSpec object containing
   *     the permissions prefix and the permissions requested.
   * @param {Object} dialogState JSON object the action uses to hold dialog state that
   *     will be circulated back by Assistant.
   * @return {Object} HTTP response object.
   * @private
   * @actionssdk
   */
  fulfillPermissionsRequest_ (permissionsSpec, dialogState) {
    debug('fulfillPermissionsRequest_: permissionsSpec=%s, dialogState=%s',
      JSON.stringify(permissionsSpec), JSON.stringify(dialogState));
    // Build an Expected Intent object.
    const expectedIntent = {
      intent: this.StandardIntents.PERMISSION,
      inputValueSpec: {
        permissionValueSpec: permissionsSpec
      }
    };
    // Send an Ask request to Assistant.
    const inputPrompt = this.buildInputPrompt(false, 'PLACEHOLDER_FOR_PERMISSION');
    return this.buildAskHelper_(inputPrompt, [expectedIntent], dialogState);
  }

  /**
   * Builds the ask response to send back to Assistant.
   *
   * @param {Object} inputPrompt Holding rich, initial and no-input prompts.
   * @param {Array} possibleIntents Array of ExpectedIntents.
   * @param {Object} dialogState JSON object the action uses to hold dialog state that
   *     will be circulated back by Assistant.
   * @return The response that is sent to Assistant to ask user to provide input.
   * @private
   * @actionssdk
   */
  buildAskHelper_ (inputPrompt, possibleIntents, dialogState) {
    debug('buildAskHelper_: inputPrompt=%s, possibleIntents=%s,  dialogState=%s',
      inputPrompt, possibleIntents, JSON.stringify(dialogState));
    if (!inputPrompt) {
      this.handleError_('Invalid input prompt');
      return null;
    }
    if (typeof inputPrompt === 'string') {
      inputPrompt = this.buildInputPrompt(this.isSsml_(inputPrompt), inputPrompt);
    } else {
      if (inputPrompt.speech) { // Check for SimpleResponse
        inputPrompt = { richInitialPrompt: this.buildRichResponse()
          .addSimpleResponse(inputPrompt) };
      } else if (inputPrompt.items) { // Check for RichResponse
        inputPrompt = { richInitialPrompt: inputPrompt };
      }
    }
    if (!dialogState) {
      dialogState = {
        'state': (this.state instanceof State ? this.state.getName() : this.state),
        'data': this.data
      };
    } else if (Array.isArray(dialogState)) {
      this.handleError_('Invalid dialog state');
      return null;
    }
    const expectedInputs = [{
      inputPrompt: inputPrompt,
      possibleIntents: possibleIntents
    }];
    const response = this.buildResponseHelper_(
      JSON.stringify(dialogState),
      true, // expectedUserResponse
      expectedInputs,
      null // finalResponse is null b/c dialog is active
    );
    return this.doResponse_(response, RESPONSE_CODE_OK);
  }

  /**
   * Builds an ExpectedIntent object. Refer to {@link ActionsSdkAction#newRuntimeEntity} to create the list
   * of runtime entities required by this method. Runtime entities need to be defined in
   * the Action Package.
   *
   * @param {string} intent Developer specified in-dialog intent inside the Action
   *     Package or an Assistant built-in intent like
   *     'action.intent.action.TEXT'.
   * @return {Object} An {@link https://developers.google.com/actions/reference/conversation#ExpectedIntent|ExpectedIntent object}
         encapsulating the intent and the runtime entities.
   * @private
   * @actionssdk
   */
  buildExpectedIntent_ (intent) {
    debug('buildExpectedIntent_: intent=%s', intent);
    if (!intent || intent === '') {
      this.handleError_('Invalid intent');
      return null;
    }
    const expectedIntent = {
      intent: intent
    };
    return expectedIntent;
  }


  /**
   * Validates whether request is from Assistant through signature verification.
   * Uses Google-Auth-Library to verify authorization token against given
   * Google Cloud Project ID. Auth token is given in request header with key,
   * "Authorization".
   *
   * @example
   * const app = new ActionsSdkApp({request, response});
   * app.isRequestFromAssistant('nodejs-cloud-test-project-1234')
   *   .then(() => {
   *     app.ask('Hey there, thanks for stopping by!');
   *   })
   *   .catch(err => {
   *     response.status(400).send();
   *   });
   *
   * @param {string} projectId Google Cloud Project ID for the Assistant app.
   * @param {Promise} Promise resolving with ID token if request is from
   *     a valid source, otherwise rejects with the error reason for an invalid
   *     token.
   */
  isRequestFromAssistant(projectId) {
    debug('isRequestFromAssistant: projectId=%s', projectId);
    const googleAuthClient = require('./utils/auth').googleAuthClient;
    const jwtToken = this.request_.get(CONVERSATION_API_SIGNATURE_HEADER);
    return new Promise((resolve, reject) => {
      if (!jwtToken) {
        const errorMsg = 'No incoming API Signature JWT token';
        error(errorMsg);
        return reject(errorMsg);
      }
      googleAuthClient.verifyIdToken(jwtToken, projectId, (err, idToken) => {
        if (err) {
          error('ID token verification Failed: ' + err);
          return reject(err);
        } else {
          return resolve(idToken);
        }
      });
    });
  }
};

module.exports = ActionsSdkAction;

