// Copyright 2016, Google, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

'use strict';

process.env.DEBUG = 'actions-on-google:*';

let express = require('express');
let bodyParser = require('body-parser');
let FoodOrderingActions = require('./actions-on-google').FoodOrderingActions;
let constants = require('./constants.json');
// Id of the order to be updated. Used for sending async order update to Google.
// Can be modified with a SubmitOrder request.
let googleOrderId = constants.testGoogleOrderID;
let actionOrderId = constants.testActionOrderID;

let app = express();
app.use(express.static('public'));
app.use(bodyParser.json({type: 'application/json'}));

/**
 * Class representing a certain amount of currency.
 */
const Money = class {
  /**
   * Constructor for class Money.
   * currencyCode : The short code for the currency in which prices are quoted.
   * units : The whole number units of currency involved.
   * nanos : The fractional units of currency. For example: 1.75$ is represented
   *         as 1 unit and 750,000,000 nanos.
   */
  constructor(money) {
    this.currencyCode = 'USD';
    this.units = 0;
    this.nanos = 0;

    if (money) {
      if (money.currencyCode) {
        this.currencyCode = money.currencyCode;
      }
      if (money.units) {
        this.units = money.units;
      }
      if (money.nanos) {
        this.nanos = money.nanos;
      }
    }
  }

  /**
   * Adds the given money values to this class.
   */
  add(currencyCode, units, nanos) {
    if (units) {
      this.units += parseInt(units);
    }
    if (nanos) {
      this.units += Math.floor((this.nanos + nanos) / 1000000000);
      this.nanos = (this.nanos + nanos) % 1000000000;
    }
  }

  /**
   * Returns the currency, units and nanos in this class.
   * @return {Object} Object of class Money.
   */
  get() {
    return {
      'currencyCode': this.currencyCode,
      'units': this.units,
      'nanos': this.nanos
    };
  }
};

app.post('/', function(req, res) {
  const assistant = new FoodOrderingActions({request: req, response: res});
  function mainIntent(assistant) {
    console.log('mainIntent');
    let inputPrompt = assistant.buildInputPrompt(
        true,
        '<speak>' +
            'I can order food. Say as <break time="1"/>Order food from oren hummus.</speak>',
        ['If you\'re still there, where do you want to order food from?']);
    assistant.ask(inputPrompt);
  }

  // Handler for CHECKOUT intent
  function checkout(assistant) {
    console.log('actions.foodordering.intent.CHECKOUT');
    const foodOrderError =
        assistant.validateCheckoutRequest(assistant.getTopInput_());
    if (foodOrderError) {
      assistant.doErrorResponse(foodOrderError);
      return null;
    }

    const cart = assistant.getCart();

    // Calculate total item price
    let totalPrice = new Money();
    if (cart.lineItems) {
      for (let item of cart.lineItems) {
        let itemPrice = item.price;
        if (itemPrice && itemPrice.amount) {
          totalPrice.add(
              itemPrice.amount.currencyCode, itemPrice.amount.units,
              itemPrice.amount.nanos);
        }
      }
    }
    // Calculate fees
    let deliveryFees = assistant.buildLineItem('Delivery Fees')
                           .setPrice('ESTIMATE', 'USD', '3', 500000000)
                           .setType(assistant.Transactions.ItemType.DELIVERY);
    totalPrice.add('USD', 3, 500000000);

    let tax = assistant.buildLineItem('Tax')
                  .setPrice('ESTIMATE', 'USD', '1', 370000000)
                  .setType(assistant.Transactions.ItemType.TAX);
    totalPrice.add('USD', 1, 370000000);

    // Build the order based on the cart
    const order =
        assistant.buildProposedOrder(cart, totalPrice, deliveryFees, tax);

    /* Example action facilitated transaction config
    const actionTransactionConfig = {
      type: assistant.Transactions.OtherPaymentType.PAYMENT_CARD,
      displayName: 'VISA-1234',
      shippingAddressRequired: false
    };*/

    // Example google processed transaction live config. This was generated by
    // creating an account on Stripe.com, activating it and then picking the
    // live key values from their API dashboard.
    const googleTransactionConfig = {
      tokenizationType: 'PAYMENT_GATEWAY',
      tokenizationParameters: {
        'gateway': 'stripe',
        'stripe:publishableKey': 'pk_live_GGZq3hM6cddZM1H52ABLxvSa',
        'stripe:version': '2017-04-06'
      },
      // As defined in assistant.Transactions.CardNetwork.
      supportedCardNetworks: ['AMEX', 'DISCOVER', 'MASTERCARD', 'VISA'],
      prepaidCardDisallowed: true
    };

    const response =
        assistant.buildCheckoutResponse(order, googleTransactionConfig);
    assistant.doDirectActionResponse(response);
  }

  // Handler for TRANSACTION_DECISION intent
  function submitOrder(assistant) {
    console.log('TRANSACTION_DECISION');
    const isValid =
        assistant.validateSubmitOrderRequest(assistant.getTopInput_());
    if (!isValid) {
      return null;
    }
    const response = assistant.buildTransactionResponse();
    // Action order id will be used for sending subsequent async order updates.
    actionOrderId = response.orderUpdate.actionOrderId;
    assistant.doDirectActionResponse(response);
  }

  let actionMap = new Map();
  actionMap.set(assistant.StandardIntents.MAIN, mainIntent);
  actionMap.set('actions.foodordering.intent.CHECKOUT', checkout);
  actionMap.set(assistant.StandardIntents.TRANSACTION_DECISION, submitOrder);

  /** Request verification
  assistant.isRequestFromAssistant(constants.agentProjectId)
      .then(() => {
        assistant.handleRequest(actionMap);
      })
      .catch(err => {
        res.status(400).send(err);
      });
   */
  
   assistant.handleRequest(actionMap);
});

app.get('/orderupdate', function(req, res) {
  const assistant = new FoodOrderingActions({request: req, response: res});
  // Send a confirmed order update for the latest placed order.
  assistant.sendOrderUpdateRequestAsync(actionOrderId, 'CONFIRMED')
      .then(
          function(success) {
            console.log(success);
            res.send('SUCCESS:Sent update. ' + success);
          },
          function(error) {
            console.log(error);
            res.status(400).send('FAILED: ' + error);
          });
});

if (module === require.main) {
  // [START server]
  // Start the server
  let server = app.listen(process.env.PORT || 8080, function() {
    let port = server.address().port;
    console.log('App listening on port %s', port);
  });
  // [END server]
}

module.exports = app;

