/**
 * Copyright 2016 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The Actions on Google client library Action base class.
 *
 * This class contains the methods that are shared between platforms to support
 * the conversation API protocol form Action. It also exports the 'State' class
 * as a helper to represent states by name.
 */

'use strict';

const Debug = require('debug');
const debug = Debug('actions-on-google:debug');
const error = Debug('actions-on-google:error');

const transformToSnakeCase = require('./utils/transform').transformToSnakeCase;
const transformToCamelCase = require('./utils/transform').transformToCamelCase;

// Constants
const ERROR_MESSAGE = 'Sorry, I am unable to process your request.';
const API_ERROR_MESSAGE_PREFIX = 'Action Error: ';
const CONVERSATION_API_VERSION_HEADER = 'Google-Assistant-API-Version';
const ACTIONS_CONVERSATION_API_VERSION_HEADER = 'Google-Actions-API-Version';
const ACTIONS_CONVERSATION_API_VERSION_TWO = 2;
const RESPONSE_CODE_OK = 200;
const RESPONSE_CODE_BAD_REQUEST = 400;
const HTTP_CONTENT_TYPE_HEADER = 'Content-Type';
const HTTP_CONTENT_TYPE_JSON = 'application/json';

// Configure logging for hosting platforms that only support console.log and console.error
debug.log = console.log.bind(console);
error.log = console.error.bind(console);

/**
 * Constructor for Action object.
 * Should not be instantiated; rather instantiate one of the subclasses
 * {@link ActionsSdkAction} or {@link ApiAiAction}.
 *
 * @param {Object} options JSON configuration.
 * @param {Object} options.request Express HTTP request object.
 * @param {Object} options.response Express HTTP response object.
 * @param {Function=} options.sessionStarted Function callback when session starts.
 */
const Action = class {
  constructor (options) {
    debug('Action constructor');

    if (!options) {
      // ignore for JavaScript inheritance to work
      return;
    }
    if (!options.request) {
      this.handleError_('Request can NOT be empty.');
      return;
    }
    if (!options.response) {
      this.handleError_('Response can NOT be empty.');
      return;
    }

    /**
     * The Express HTTP request that the endpoint receives from the Assistant.
     * @private
     * @type {Object}
     */
    this.request_ = options.request;

    /**
     * The Express HTTP response the endpoint will return to Assistant.
     * @private
     * @type {Object}
     */
    this.response_ = options.response;

    /**
     * 'sessionStarted' callback (optional).
     * @private
     * @type {Function}
     */
    this.sessionStarted_ = options.sessionStarted;

    debug('Request from Assistant: %s', JSON.stringify(this.request_.body));

    /**
     * The request body contains query JSON and previous session variables.
     * Assignment using JSON parse/stringify ensures manipulation of this.body_
     * does not affect passed in request body structure.
     * @private
     * @type {Object}
     */
    this.body_ = JSON.parse(JSON.stringify(this.request_.body));

    /**
     * API version describes version of the Actions API request.
     * @private
     * @type {string}
     */
    this.actionsApiVersion_ = null;
    // Populates API version from either request header or APIAI orig request.
    if (this.request_.get(ACTIONS_CONVERSATION_API_VERSION_HEADER)) {
      this.actionsApiVersion_ = this.request_.get(ACTIONS_CONVERSATION_API_VERSION_HEADER);
      debug('Actions API version from header: ' + this.actionsApiVersion_);
    }
    if (this.body_.originalRequest &&
      this.body_.originalRequest.version) {
      this.actionsApiVersion_ = this.body_.originalRequest.version;
      debug('Actions API version from APIAI: ' + this.actionsApiVersion_);
    }

    // If request is in Proto2 format, convert to Proto3
    if (!this.isProto3_()) {
      if (this.body_.originalRequest) {
        this.body_.originalRequest = transformToCamelCase(this.body_.originalRequest);
      } else {
        this.body_ = transformToCamelCase(this.body_);
      }
    }

    /**
     * Intent handling data structure.
     * @private
     * @type {Object}
     */
    this.handler_ = null;

    /**
     * Intent mapping data structure.
     * @private
     * @type {Object}
     */
    this.intentMap_ = null;

    /**
     * Intent state data structure.
     * @private
     * @type {Object}
     */
    this.stateMap_ = null;

    /**
     * The session state.
     * @public
     * @type {string}
     */
    this.state = null;

    /**
     * The session data in JSON format.
     * @public
     * @type {Object}
     */
    this.data = {};

    /**
     * The API.AI context.
     * @private
     * @type {Object}
     */
    this.contexts_ = {};

    /**
     * The last error message.
     * @private
     * @type {string}
     */
    this.lastErrorMessage_ = null;

    /**
     * Track if an HTTP response has been sent already.
     * @private
     * @type {boolean}
     */
    this.responded_ = false;

    /**
     * List of standard intents that the Action provides.
     * @readonly
     * @enum {string}
     * @actionssdk
     * @apiai
     */
    this.StandardIntents = {
      /** Assistant fires MAIN intent for queries like [talk to $action]. */
      MAIN: this.isProto3_() ? 'actions.intent.MAIN' : 'assistant.intent.action.MAIN',
      /** Assistant fires TEXT intent when action issues ask intent. */
      TEXT: this.isProto3_() ? 'actions.intent.TEXT' : 'assistant.intent.action.TEXT',
      /** Assistant fires PERMISSION intent when action invokes askForPermission. */
      PERMISSION: this.isProto3_() ? 'actions.intent.PERMISSION' : 'assistant.intent.action.PERMISSION',
      /** Action fires OPTION intent when action elicits selection from user. */
      OPTION: 'actions.intent.OPTION',
      /** Action fires TRANSACTION_REQUIREMENTS_CHECK intent when action sets up transaction. */
      TRANSACTION_REQUIREMENTS_CHECK: 'actions.intent.TRANSACTION_REQUIREMENTS_CHECK',
      /** Action fires DELIVERY_ADDRESS intent when action asks for delivery address. */
      DELIVERY_ADDRESS: 'actions.intent.DELIVERY_ADDRESS',
      /** Action fires TRANSACTION_DECISION intent when action asks for transaction decision. */
      TRANSACTION_DECISION: 'actions.intent.TRANSACTION_DECISION'
    };

    /**
     * List of supported permissions the Action supports.
     * @readonly
     * @enum {string}
     * @actionssdk
     * @apiai
     */
    this.SupportedPermissions = {
      /**
       * The user's name as defined in the
       * {@link https://developers.google.com/actions/reference/conversation#UserProfile|UserProfile object}
       */
      NAME: 'NAME',
      /**
       * The location of the user's current device, as defined in the
       * {@link https://developers.google.com/actions/reference/conversation#Location|Location object}.
       */
      DEVICE_PRECISE_LOCATION: 'DEVICE_PRECISE_LOCATION',
      /**
       * City and zipcode corresponding to the location of the user's current device, as defined in the
       * {@link https://developers.google.com/actions/reference/conversation#Location|Location object}.
       */
      DEVICE_COARSE_LOCATION: 'DEVICE_COARSE_LOCATION'
    };

    /**
     * List of built-in argument names.
     * @readonly
     * @enum {string}
     * @actionssdk
     * @apiai
     */
    this.BuiltInArgNames = {
      /** Permission granted argument. */
      PERMISSION_GRANTED: this.isProto3_() ? 'PERMISSION' : 'permission_granted'
    };

    /**
     * List of possible conversation stages, as defined in the
     * {@link https://developers.google.com/actions/reference/conversation#Conversation|Conversation object}.
     * @readonly
     * @enum {number}
     * @actionssdk
     * @apiai
     */
    this.ConversationStages = {
      /**
       * Unspecified conversation state.
       */
      UNSPECIFIED: this.isProto3_() ? 'UNSPECIFIED' : 0,
      /**
       * A new conversation.
       */
      NEW: this.isProto3_() ? 'NEW' : 1,
      /**
       * An active (ongoing) conversation.
       */
      ACTIVE: this.isProto3_() ? 'ACTIVE' : 2
    };

    /**
     * List of supported surface capabilities the Action supports.
     * @readonly
     * @enum {string}
     * @actionssdk
     * @apiai
     */
    this.SurfaceCapabilities = {
      /**
       * The ability to output audio.
       */
      AUDIO_OUTPUT: 'actions.capability.AUDIO_OUTPUT',
      /**
       * The ability to output on a screen
       */
      SCREEN_OUTPUT: 'actions.capability.SCREEN_OUTPUT'
    };

    /**
     * List of possible user input types.
     * @readonly
     * @enum {number}
     * @actionssdk
     * @apiai
     */
    this.InputTypes = {
      /**
       * Unspecified.
       */
      UNSPECIFIED: this.isProto3_() ? 'UNSPECIFIED' : 0,
      /**
       * Input given by touch.
       */
      TOUCH: this.isProto3_() ? 'TOUCH' : 1,
      /**
       * Input given by voice (spoken).
       */
      VOICE: this.isProto3_() ? 'VOICE' : 2,
      /**
       * Input given by keyboard (typed).
       */
      KEYBOARD: this.isProto3_() ? 'KEYBOARD' : 3
    };

    /**
     * Values related to supporting transactions
     * @readonly
     * @enum {string}
     * @actionssdk
     * @apiai
     */
    this.Transactions = {
      /**
       * List of transaction card networks available when paying with Google.
       */
      CardNetwork: {
        /**
         * Unspecified.
         */
        UNSPECIFIED: this.isProto3_() ? 'UNSPECIFIED' : 0,
        /**
         * American Express.
         */
        AMEX: this.isProto3_() ? 'AMEX' : 1,
        /**
         * Discover.
         */
        DISCOVER: this.isProto3_() ? 'DISCOVER' : 2,
        /**
         * Master Card.
         */
        MASTERCARD: this.isProto3_() ? 'MASTERCARD' : 3,
        /**
         * Visa.
         */
        VISA: this.isProto3_() ? 'VISA' : 4,
        /**
         * JCB.
         */
        JCB: this.isProto3_() ? 'JCB' : 5
      },
      /**
       * List of possible item types.
       */
      ItemType: {
        /**
         * Unspecified.
         */
        UNSPECIFIED: 'UNSPECIFIED',
        /**
         * Regular.
         */
        REGULAR: 'REGULAR',
        /**
         * Tax.
         */
        TAX: 'TAX',
        /**
         * Discount
         */
        DISCOUNT: 'DISCOUNT',
        /**
         * Gratuity
         */
        GRATUITY: 'GRATUITY',
        /**
         * Delivery
         */
        DELIVERY: 'DELIVERY',
        /**
         * Subtotal
         */
        SUBTOTAL: 'SUBTOTAL',
        /**
         * Fee. For everything else, there's fee.
         */
        FEE: 'FEE'
      },
      /**
       * List of price types.
       */
      PriceType: {
        /**
         * Unknown.
         */
        UNKNOWN: 'UNKNOWN',
        /**
         * Estimate.
         */
        ESTIMATE: 'ESTIMATE',
        /**
         * Actual.
         */
        ACTUAL: 'ACTUAL'
      },
      /**
       * List of possible item types.
       */
      PaymentType: {
        /**
         * Unspecified.
         */
        UNSPECIFIED: 'UNSPECIFIED',
        /**
         * Payment card.
         */
        PAYMENT_CARD: 'PAYMENT_CARD',
        /**
         * Bank.
         */
        BANK: 'BANK',
        /**
         * Loyalty program.
         */
        LOYALTY_PROGRAM: 'LOYALTY_PROGRAM',
        /**
         * On order fulfilment, such as cash on delivery.
         */
        ON_FULFILLMENT: 'ON_FULFILLMENT',
        /**
         * Gift card.
         */
        GIFT_CARD: 'GIFT_CARD'
      },
      /**
       * List of possible order confirmation user decisions
       */
      ConfirmationDecision: {
        /**
         * Order was approved by user.
         */
        ACCEPTED: this.isProto3_() ? 'ORDER_ACCEPTED' : 1,
        /**
         * Order was declined by user.
         */
        REJECTED: this.isProto3_() ? 'ORDER_REJECTED' : 2,
        /**
         * Order was not declined, but the delivery address was updated during
         * confirmation.
         */
        DELIVERY_ADDRESS_UPDATED: this.isProto3_() ? 'DELIVERY_ADDRESS_UPDATED' : 3,
        /**
         * Order was not declined, but the cart was updated during
         * confirmation.
         */
        CART_CHANGE_REQUESTED: this.isProto3_() ? 'CART_CHANGE_REQUESTED' : 4
      },
      /**
       * List of possible order states.
       */
      OrderState: {
        /**
         * Order was confirmed by user.
         */
        CREATED: 'CREATED',
        /**
         * Order was rejected.
         */
        REJECTED: 'REJECTED',
        /**
         * Order was confirmed by integrator and is active.
         */
        CONFIRMED: 'CONFIRMED',
        /**
         * User cancelled the order.
         */
        CANCELLED: 'CANCELLED',
        /**
         * Order is being delivered.
         */
        IN_TRANSIT: 'IN_TRANSIT',
        /**
         * User performed a return.
         */
        RETURNED: 'RETURNED',
        /**
         * User received what was ordered.
         */
        FULFILLED: 'FULFILLED'
      },
      /**
       * List of possible actions to take on the order.
       */
      OrderAction: {
        /**
         * View details.
         */
        VIEW_DETAILS: 'VIEW_DETAILS',
        /**
         * Modify order.
         */
        MODIFY: 'MODIFY',
        /**
         * Cancel order.
         */
        CANCEL: 'CANCEL',
        /**
         * Return order.
         */
        RETURN: 'RETURN',
        /**
         * Exchange order.
         */
        EXCHANGE: 'EXCHANGE',
        /**
         * Email.
         */
        EMAIL: 'EMAIL',
        /**
         * Call.
         */
        CALL: 'CALL',
        /**
         * Reorder.
         */
        REORDER: 'REORDER',
        /**
         * Review.
         */
        REVIEW: 'REVIEW'
      },
      /**
       * List of possible types of order rejection.
       */
      RejectionType: {
        /**
         * Unknown
         */
        UNKNOWN: 'UNKNOWN',
        /**
         * Payment was declined.
         */
        PAYMENT_DECLINED: 'PAYMENT_DECLINED'
      }
    };

    /**
     * API version describes version of the Assistant request.
     * @deprecated
     * @private
     * @type {string}
     */
    this.apiVersion_ = null;
    // Populates API version.
    if (this.request_.get(CONVERSATION_API_VERSION_HEADER)) {
      this.apiVersion_ = this.request_.get(CONVERSATION_API_VERSION_HEADER);
      debug('Assistant API version: ' + this.apiVersion_);
    }
  }

  // ---------------------------------------------------------------------------
  //                   Public APIs
  // ---------------------------------------------------------------------------

  /**
   * Handles the incoming Assistant request using a handler or Map of handlers.
   * Each handler can be a function callback or Promise.
   *
   * @example
   * // Actions SDK
   * const action = new ActionsSdkAction({request: request, response: response});
   *
   * function mainIntent (action) {
   *   const inputPrompt = action.buildInputPrompt(true, '<speak>Hi! <break time="1"/> ' +
   *         'I can read out an ordinal like ' +
   *         '<say-as interpret-as="ordinal">123</say-as>. Say a number.</speak>',
   *         ['I didn\'t hear a number', 'If you\'re still there, what\'s the number?', 'What is the number?']);
   *   action.ask(inputPrompt);
   * }
   *
   * function rawInput (action) {
   *   if (action.getRawInput() === 'bye') {
   *     action.tell('Goodbye!');
   *   } else {
   *     const inputPrompt = action.buildInputPrompt(true, '<speak>You said, <say-as interpret-as="ordinal">' +
   *       action.getRawInput() + '</say-as></speak>',
   *         ['I didn\'t hear a number', 'If you\'re still there, what\'s the number?', 'What is the number?']);
   *     action.ask(inputPrompt);
   *   }
   * }
   *
   * const actionMap = new Map();
   * actionMap.set(action.StandardIntents.MAIN, mainIntent);
   * actionMap.set(action.StandardIntents.TEXT, rawInput);
   *
   * action.handleRequest(actionMap);
   *
   * // API.AI
   * const action = new ApiAiAction({request: req, response: res});
   * const NAME_ACTION = 'make_name';
   * const COLOR_ARGUMENT = 'color';
   * const NUMBER_ARGUMENT = 'number';
   *
   * function makeName (action) {
   *   const number = action.getArgument(NUMBER_ARGUMENT);
   *   const color = action.getArgument(COLOR_ARGUMENT);
   *   action.tell('Alright, your silly name is ' +
   *     color + ' ' + number +
   *     '! I hope you like it. See you next time.');
   * }
   *
   * const actionMap = new Map();
   * actionMap.set(NAME_ACTION, makeName);
   * action.handleRequest(actionMap);
   *
   * @param {(Function|Map)} handler The handler (or Map of handlers) for the request.
   * @actionssdk
   * @apiai
   */
  handleRequest (handler) {
    debug('handleRequest: handler=%s', handler);
    if (!handler) {
      this.handleError_('request handler can NOT be empty.');
      return;
    }
    this.extractData_();
    if (typeof handler === 'function') {
      debug('handleRequest: function');
      // simple function handler
      this.handler_ = handler;
      const promise = handler(this);
      if (promise instanceof Promise) {
        promise.then(
          (result) => {
            debug(result);
          })
        .catch(
          (reason) => {
            this.handleError_('function failed: %s', reason.message);
            this.tell(!reason.message ? ERROR_MESSAGE : reason.message);
          });
      } else {
        // Handle functions
        return;
      }
      return;
    } else if (handler instanceof Map) {
      debug('handleRequest: map');
      const intent = this.getIntent();
      const result = this.invokeIntentHandler_(handler, intent);
      if (!result) {
        this.tell(!this.lastErrorMessage_ ? ERROR_MESSAGE : this.lastErrorMessage_);
      }
      return;
    }
    // Could not handle intent
    this.handleError_('invalid intent handler type: ' + (typeof handler));
    this.tell(ERROR_MESSAGE);
  }

  /**
   * Equivalent to {@link Action#askForPermission|askForPermission},
   * but allows you to prompt the user for more than one permission at once.
   *
   * Notes:
   *
   * * The order in which you specify the permission prompts does not matter -
   *   it is controlled by the action to provide a consistent user experience.
   * * The user will be able to either accept all permissions at once, or none.
   *   If you wish to allow them to selectively accept one or other, make several
   *   dialog turns asking for each permission independently with askForPermission.
   * * Asking for DEVICE_COARSE_LOCATION and DEVICE_PRECISE_LOCATION at once is
   *   equivalent to just asking for DEVICE_PRECISE_LOCATION
   *
   * @example
   * const action = new ApiAiAction({request: req, response: res});
   * const REQUEST_PERMISSION_ACTION = 'request_permission';
   * const GET_RIDE_ACTION = 'get_ride';
   *
   * function requestPermission (action) {
   *   const permission = [
   *     action.SupportedPermissions.NAME,
   *     action.SupportedPermissions.DEVICE_PRECISE_LOCATION
   *   ];
   *   action.askForPermissions('To pick you up', permissions);
   * }
   *
   * function sendRide (action) {
   *   if (action.isPermissionGranted()) {
   *     const displayName = action.getUserName().displayName;
   *     const address = action.getDeviceLocation().address;
   *     action.tell('I will tell your driver to pick up ' + displayName +
   *         ' at ' + address);
   *   } else {
   *     // Response shows that user did not grant permission
   *     action.tell('Sorry, I could not figure out where to pick you up.');
   *   }
   * }
   * const actionMap = new Map();
   * actionMap.set(REQUEST_PERMISSION_ACTION, requestPermission);
   * actionMap.set(GET_RIDE_ACTION, sendRide);
   * action.handleRequest(actionMap);
   *
   * @param {string} context Context why the permission is being asked; it's the TTS
   *     prompt prefix (action phrase) we ask the user.
   * @param {Array<string>} permissions Array of permissions Action supports, each of
   *     which comes from Action.SupportedPermissions.
   * @param {Object=} dialogState JSON object the action uses to hold dialog state that
   *     will be circulated back by Assistant.
   * @return A response is sent to Assistant to ask for the user's permission; for any
   *     invalid input, we return null.
   * @actionssdk
   * @apiai
   */
  askForPermissions (context, permissions, dialogState) {
    debug('askForPermissions: context=%s, permissions=%s, dialogState=%s',
      context, permissions, JSON.stringify(dialogState));
    if (!context || context === '') {
      this.handleError_('Assistant context can NOT be empty.');
      return null;
    }
    if (!permissions || permissions.length === 0) {
      this.handleError_('At least one permission needed.');
      return null;
    }
    for (let i = 0; i < permissions.length; i++) {
      const permission = permissions[i];
      if (permission !== this.SupportedPermissions.NAME &&
        permission !== this.SupportedPermissions.DEVICE_PRECISE_LOCATION &&
        permission !== this.SupportedPermissions.DEVICE_COARSE_LOCATION) {
        this.handleError_('Action permission must be one of ' +
          '[NAME, DEVICE_PRECISE_LOCATION, DEVICE_COARSE_LOCATION]');
        return null;
      }
    }
    if (!dialogState) {
      dialogState = {
        'state': (this.state instanceof State ? this.state.getName() : this.state),
        'data': this.data
      };
    }
    return this.fulfillPermissionsRequest_({
      optContext: context,
      permissions: permissions
    }, dialogState);
  }

  /**
   * Asks the Assistant to guide the user to grant a permission. For example,
   * if you want your action to get access to the user's name, you would invoke
   * the askForPermission method with a context containing the reason for the request,
   * and the action.SupportedPermissions.NAME permission. With this, the Assistant will ask
   * the user, in your agent's voice, the following: '[Context with reason for the request],
   * I'll just need to get your name from Google, is that OK?'.
   *
   * Once the user accepts or denies the request, the Assistant will fire another intent:
   * action.intent.action.PERMISSION with a boolean argument: action.BuiltInArgNames.PERMISSION_GRANTED
   * and, if granted, the information that you requested.
   *
   * Read more:
   *
   * * {@link https://developers.google.com/actions/reference/conversation#ExpectedIntent|Supported Permissions}
   * * Check if the permission has been granted with {@link ActionsSdkAction#isPermissionGranted|isPermissionsGranted}
   * * {@link ActionsSdkAction#getDeviceLocation|getDeviceLocation}
   * * {@link Action#getUserName|getUserName}
   *
   * @example
   * const action = new ApiAiAction({request: req, response: res});
   * const REQUEST_PERMISSION_ACTION = 'request_permission';
   * const GET_RIDE_ACTION = 'get_ride';
   *
   * function requestPermission (action) {
   *   const permission = action.SupportedPermissions.NAME;
   *   action.askForPermission('To pick you up', permission);
   * }
   *
   * function sendRide (action) {
   *   if (action.isPermissionGranted()) {
   *     const displayName = action.getUserName().displayName;
   *     action.tell('I will tell your driver to pick up ' + displayName);
   *   } else {
   *     // Response shows that user did not grant permission
   *     action.tell('Sorry, I could not figure out who to pick up.');
   *   }
   * }
   * const actionMap = new Map();
   * actionMap.set(REQUEST_PERMISSION_ACTION, requestPermission);
   * actionMap.set(GET_RIDE_ACTION, sendRide);
   * action.handleRequest(actionMap);
   *
   * @param {string} context Context why permission is asked; it's the TTS
   *     prompt prefix (action phrase) we ask the user.
   * @param {string} permission One of the permissions Assistant supports, each of
   *     which comes from Action.SupportedPermissions.
   * @param {Object=} dialogState JSON object the action uses to hold dialog state that
   *     will be circulated back by Assistant.
   * @return A response is sent to the Assistant to ask for the user's permission;
   *     for any invalid input, we return null.
   * @actionssdk
   * @apiai
   */
  askForPermission (context, permission, dialogState) {
    debug('askForPermission: context=%s, permission=%s, dialogState=%s',
      context, permission, JSON.stringify(dialogState));
    return this.askForPermissions(context, [permission], dialogState);
  }

  /**
   * User's permissioned name info.
   * @typedef {Object} UserName
   * @property {string} displayName - User's display name.
   * @property {string} givenName - User's given name.
   * @property {string} familyName - User's family name.
   */

  /**
   * User's permissioned device location.
   * @typedef {Object} DeviceLocation
   * @property {Object} coordinates - {latitude, longitude}. Requested with
   *     SupportedPermissions.DEVICE_PRECISE_LOCATION.
   * @property {string} address - Full, formatted street address. Requested with
   *     SupportedPermissions.DEVICE_PRECISE_LOCATION.
   * @property {string} zipCode - Zip code. Requested with
   *      SupportedPermissions.DEVICE_COARSE_LOCATION.
   * @property {string} city - Device city. Requested with
   *     SupportedPermissions.DEVICE_COARSE_LOCATION.
   */

   /**
   * User object.
   * @typedef {Object} User
   * @property {string} userId - Random string ID for Google user.
   * @property {UserName} userName - User name information. Null if not
   *     requested with {@link Action#askForPermission|askForPermission(SupportedPermissions.NAME)}.
   * @property {string} accessToken - Unique Oauth2 token. Only available with
   *     account linking.
   */

  /**
   * If granted permission to user's name in previous intent, returns user's
   * display name, family name, and given name. If name info is unavailable,
   * returns null.
   *
   * @example
   * const action = new ApiAiAction({request: req, response: res});
   * const REQUEST_PERMISSION_ACTION = 'request_permission';
   * const SAY_NAME_ACTION = 'get_name';
   *
   * function requestPermission (action) {
   *   const permission = action.SupportedPermissions.NAME;
   *   action.askForPermission('To know who you are', permission);
   * }
   *
   * function sayName (action) {
   *   if (action.isPermissionGranted()) {
   *     action.tell('Your name is ' + action.getUserName().displayName));
   *   } else {
   *     // Response shows that user did not grant permission
   *     action.tell('Sorry, I could not get your name.');
   *   }
   * }
   * const actionMap = new Map();
   * actionMap.set(REQUEST_PERMISSION_ACTION, requestPermission);
   * actionMap.set(SAY_NAME_ACTION, sayName);
   * action.handleRequest(actionMap);
   * @return {UserName} Null if name permission is not granted.
   * @actionssdk
   * @apiai
   */
  getUserName () {
    debug('getUserName');
    return this.getUser().userName;
  }

  // ---------------------------------------------------------------------------
  //                   Multimodal Builders
  // ---------------------------------------------------------------------------

  // Build RichResponse object with chainable methods
  buildRichResponse () {
    const that = this;
    const richResponse = {
      items: [],
      suggestions: [],
      // Takes string, or { speech, display_text }
      addSimpleResponse: function (simpleResponse) {
        // Validate if RichResponse already contains two SimpleResponse objects
        let simpleResponseCount = 0;
        for (let item of this.items) {
          if (item.simpleResponse) {
            simpleResponseCount++;
          }
          if (simpleResponseCount >= 2) {
            debug('Cannot include >2 SimpleResponses in RichResponse');
            return this;
          }
        }
        const simpleResponseObj = {
          simpleResponse: that.buildSimpleResponseHelper_(simpleResponse)
        };
        // Check first if needs to replace BasicCard at beginning of items list
        if (this.items.length > 0 && this.items[0].basicCard) {
          this.items.unshift(simpleResponseObj);
        } else {
          this.items.push(simpleResponseObj);
        }
        return this;
      },
      // Takes BasicCard
      addBasicCard: function (basicCard) {
        // Validate if basic card is already present
        for (let item of this.items) {
          if (item.basicCard) {
            debug('Cannot include >1 BasicCard in RichResponse');
            return this;
          }
        }
        this.items.push({
          basicCard: basicCard
        });
        return this;
      },
      // Takes a string or array of strings of suggestions
      addSuggestions: function (suggestions) {
        if (Array.isArray(suggestions)) {
          for (let suggestion of suggestions) {
            this.suggestions.push({title: suggestion});
          }
        } else {
          this.suggestions.push({title: suggestions});
        }
        return this;
      },
      // Takes two strings
      addSuggestionLink: function (destinationName, suggestionUrl) {
        this.linkOutSuggestion = { destinationName: destinationName, url: suggestionUrl };
        return this;
      },
      addStructuredResponse: function(structuredResponse) {
        // Validate if RichResponse already contains StructuredResponse object
        for (let item of this.items) {
          if (item.structuredResponse) {
            debug('Cannot include >2 SimpleResponses in RichResponse');
            return this;
          }
        }
        this.items.unshift({
          structuredResponse: structuredResponse
        });
        return this;
      },
      addOrderUpdate: function (orderUpdate) {
        // Validate if RichResponse already contains StructuredResponse object
        for (let item of this.items) {
          if (item.structuredResponse) {
            debug('Cannot include >2 StructuredResponses in RichResponse');
            return this;
          }
        }
        this.items.unshift({
          structuredResponse: {
            orderUpdate: orderUpdate
          }
        });
        return this;
      }

    };

    return richResponse;
  }

  // Takes req. bodyText string
  buildBasicCard (bodyText) {
    const basicCard = {
      formattedText: bodyText,
      buttons: [],
      // Takes title string
      setTitle: function (title) {
        this.title = title;
        return this;
      },
      // Takes subtitle string
      setSubtitle: function (subtitle) {
        this.subtitle = subtitle;
        return this;
      },
      // Takes image Url string, accessibility text string
      setImage: function (url, accessibilityText) {
        this.image = { url: url, accessibilityText: accessibilityText };
        return this;
      },
      // Takes text and image url strings
      addButton: function (text, url) {
        this.buttons.push({
          title: text,
          openUrlAction: {
            url: url
          }
        });
        return this;
      }
    };

    return basicCard;
  }

  // Takes opt. title string
  // Could also pass in array of items
  buildList (param) {
    const list = {
      title: Array.isArray(param) ? null : param,
      items: Array.isArray(param) ? param : [],
      // Takes item built with buildSelectionItem
      addItem: function (item) {
        this.items.push(item);
        return this;
      }
    };

    return list;
  }

  buildCarousel (param) {
    const list = {
      items: Array.isArray(param) ? param : [],
      // Takes item built with buildSelectionItem
      addItem: function (item) {
        this.items.push(item);
        return this;
      }
    };

    return list;
  }

  // Takes req. key, synonyms, title
  buildSelectionItem (key, synonyms, title) {
    const listItem = {
      optionInfo: {
        key: key,
        synonyms: synonyms
      },
      title: title,
      // Takes string
      setDescription: function (description) {
        this.description = description;
        return this;
      },
      // Takes string
      setImage: function (imageUrl, accessibilityText) {
        this.image = { url: imageUrl, accessibilityText: accessibilityText };
        return this;
      },
      // Takes string
      addSynonym: function (synonym) {
        this.optionInfo.synonyms.push(synonym);
        return this;
      }
    };

    return listItem;
  }

  // ---------------------------------------------------------------------------
  //                   Transaction Builders
  // ---------------------------------------------------------------------------

  // Build Order object with chainable methods.
  buildOrder() {
    const that = this;
    const order = {
      otherItems: [],
      // Takes Cart object
      setCart: function (cart) {
        this.cart = cart;
        return this;
      },
      // Takes single LineItem or array of LineItems
      addOtherItems: function (items) {
        if (Array.isArray((items))) {
          for (let item of items) {
            this.otherItems.push(item);
          }
        } else {
          this.otherItems.push(items);
        }
        return this;
      },
      // Takes req. image Url string, access text, opt. height/width
      setImage: function (url, accessibilityText, height, width) {
        this.image = { url, accessibilityText };
        if (height) {
          this.image.height = height;
        }
        if (width) {
          this.image.width = width;
        }
        return this;
      },
      // Takes TOS url
      setTOS (url) {
        this.termsOfServiceUrl = url;
        return this;
      },
      // Takes PriceType enum, currency code 3 letter string, 2 numbers (second is optional)
      setTotalPrice (type, currencyCode, units, nanos) {
        this.totalPrice = {
          type: type,
          amount: {
            currencyCode: currencyCode,
            units: units,
            nanos: nanos ? nanos : 0
          }
        };
        return this;
      },
      setExtension () {
        // TODO
        return this;
      }
    };

    return order;
  }

  // Build Cart object with chainable methods. Takes opt. id.
  buildCart (id) {
    const that = this;
    const cart = {
      id: id ? id : undefined,
      lineItems: [],
      otherItems: [],
      // Takes id, name
      setMerchant: function (id, name) {
        this.merchant = {
          id: id,
          name: name
        };
        return this;
      },
      // Takes single LineItem or array of LineItems
      addLineItems: function (items) {
        if (Array.isArray((items))) {
          for (let item of items) {
            this.lineItems.push(item);
          }
        } else {
          this.lineItems.push(items);
        }
        return this;
      },
      // Takes single LineItem or array of LineItems
      addOtherItems: function (items) {
        if (Array.isArray((items))) {
          for (let item of items) {
            this.otherItems.push(item);
          }
        } else {
          this.otherItems.push(items);
        }
        return this;
      },
      // Takes notes string
      setNotes: function (notes) {
        this.notes = notes;
        return this;
      }
    };

    return cart;
  }

  // Build LineItem object with chainable methods. Takes required name, id string.
  buildLineItem (name, id) {
    const that = this;
    const lineItem = {
      id: id,
      name: name,
      subLines: [],
      // Takes PriceType enum, currency code 3 letter string, 2 numbers (second is optional)
      setPrice (type, currencyCode, units, nanos) {
        this.price = {
          type: type,
          amount: {
            currencyCode: currencyCode,
            units: units,
            nanos: nanos ? nanos : 0
          }
        };
        return this;
      },
      // Takes either LineItem or string note
      addSubLine: function (item) {
        if (typeof item === 'string') {
          this.subLines.push({
            note: item
          });
        } else {
          this.subLines.push({
            lineItem: item
          });
        }
        return this;
      },
      // Must be one of action.Transactions.ItemType
      setType: function (type) {
        this.type = type;
        return this;
      },
      // Takes quantity number
      setQuantity: function (quantity) {
        this.quantity = quantity;
        return this;
      },
      // Takes description string
      setDescription: function (description) {
        this.description = description;
        return this;
      },
      // Takes req. image Url string, access text, opt. height/width
      setImage: function (url, accessibilityText, height, width) {
        this.image = { url, accessibilityText };
        if (height) {
          this.image.height = height;
        }
        if (width) {
          this.image.width = width;
        }
        return this;
      },
      // Takes offer ID string
      setOfferId: function (offerId) {
        this.offerId = offerId;
        return this;
      },
      setExtension () {
        // TODO
        return this;
      }
    };

    return lineItem;
  }

  // Build OrderUpdate object with chainable methods.
  // Takes required order id, and flag indicating whether is Google-provided order id.
  buildOrderUpdate (orderId, isGoogleOrderId) {
    const that = this;
    const orderUpdate = {
      // Takes one of action.Transaction.OrderStates and label string
      setOrderState: function (state, label) {
        this.orderState = {
          state: state,
          label: label
        };
        return this;
      },

      // Takes item Id string, PriceType enum, currency code 3 letter string,
      // 2 numbers (second is optional) indicating new price.
      // Also, a reason string indicating reason for change. Reason is required
      // unless already specified for line item status update for this item.
      addLineItemPriceUpdate: function (itemId, type, currencyCode, units,
                                        nanos, reason) {
        let lineItemUpdate = this.lineItemUpdates.hasOwnProperty(itemId)
          ? this.lineItemUpdates[itemId] : {
            price: {
              type: type,
              amount: {
                currencyCode: currencyCode,
                units: units,
                nanos: nanos ? nanos : 0
              }
            }
          };
        if (reason) {
          lineItemUpdate.reason = reason;
        }
        this.lineItemUpdates[itemId] = lineItemUpdate;
        return this;
      },
      // Takes item Id string, one of action.Transaction.OrderStates, user
      // visible label for update, and opt. reason. If price update has already
      // been added for this item, specifying reason here will overwrite
      // reason given there.
      addLineItemStateUpdate: function (itemId, state, label, reason) {
        let lineItemUpdate = this.lineItemUpdates.hasOwnProperty(itemId)
          ? this.lineItemUpdates[itemId] : {
            orderState: {
              state: state,
              label: label
            }
          };
        if (reason) {
          lineItemUpdate.reason = reason;
        }
        this.lineItemUpdates[itemId] = lineItemUpdate;
        return this;
      },
      // Takes UTC timestamp seconds since epoch, nanos (opt.)
      setUpdateTime: function (seconds, nanos) {
        this.updateTime = {
          seconds: seconds,
          nanos: nanos ? nanos : 0
        };
        return this;
      },
      // Takes one of action.Transactions.OrderActions, button
      // label string, and url to open when button is clicked
      addOrderManagementAction: function (type, label, url) {
        if (!this.orderManagementActions) {
          this.orderManagementActions = [];
        }
        this.orderManagementActions.push({
          type: type,
          button: {
            title: label,
            openUrlAction: {
              url: url
            }
          }
        });
        return this;
      },
      // Takes title, text strings
      setUserNotification: function (title, text) {
        this.userNotification = { title, text };
        return this;
      },
      // Takes PriceType enum, currency code 3 letter string, 2 numbers (second is optional)
      setTotalPrice (type, currencyCode, units, nanos) {
        this.totalPrice = {
          type: type,
          amount: {
            currencyCode: currencyCode,
            units: units,
            nanos: nanos ? nanos : 0
          }
        };
        return this;
      },
      // ONLY USE ONE OF THE FOLLOWING
      // Takes one of action.Transaction.RejectionTypes and reason string
      setRejection: function (type, reason) {
        this.rejectionInfo = {
          type: type,
          reason: reason
        };
        return this;
      },
      setReceipt: function (orderId) {
        this.receipt = {
          userVisibleOrderId: orderId
        };
        return this;
      },
      setCancellation: function (reason) {
        this.cancellationInfo = {
          reason: reason
        };
        return this;
      },
      // Takes seconds/nanos (opt.) utc timestamps
      setTransit: function (updateSeconds, updateNanos) {
        this.inTransitInfo = {
          updatedTime: {
            seconds: updateSeconds,
            nanos: updateNanos ? updateNanos : 0
          }
        };
        return this;
      },
      setFulfillment: function (deliverySeconds, deliveryNanos) {
        this.fulfillmentInfo = {
          deliveryTime: {
            seconds: deliverySeconds,
            nanos: deliveryNanos ? deliveryNanos : 0
          }
        };
      },
      setReturn: function (reason) {
        this.reasonInfo = {
          reason: reason
        };
        return this;
      }
    };
    if (isGoogleOrderId) {
      orderUpdate.googleOrderId = orderId;
    } else {
      orderUpdate.actionsOrderId = orderId;
    }
    return orderUpdate;
  }

  // ---------------------------------------------------------------------------
  //                   Private Helpers
  // ---------------------------------------------------------------------------

  /**
   * Utility function to invoke an intent handler.
   *
   * @param {Object} handler The handler for the request.
   * @param {string} intent The intent to handle.
   * @return {boolean} true if the handler was invoked.
   * @private
   */
  invokeIntentHandler_ (handler, intent) {
    debug('invokeIntentHandler_: handler=%s, intent=%s', handler, intent);
    this.lastErrorMessage_ = null;
    // map of intents or states
    for (let key of handler.keys()) {
      const value = handler.get(key);
      let name;
      if (key instanceof Intent) {
        debug('key is intent');
        name = key.getName();
      } else if (key instanceof State) {
        debug('key is state');
        name = key.getName();
      } else {
        debug('key is id');
        // String id
        name = key;
      }
      debug('name=' + name);
      if (value instanceof Map) {
        debug('state=' + (this.state instanceof State ? this.state.getName() : this.state));
        // map of states
        if (!this.state && name === null) {
          debug('undefined state');
          return this.invokeIntentHandler_(value, intent);
        } else if (this.state instanceof State && name === this.state.getName()) {
          return this.invokeIntentHandler_(value, intent);
        } else if (name === this.state) {
          return this.invokeIntentHandler_(value, intent);
        }
      }
      // else map of intents
      if (name === intent) {
        debug('map of intents');
        const promise = value(this);
        if (promise instanceof Promise) {
          promise.then(
            (result) => {
              // No-op
            })
          .catch(
            (reason) => {
              error(reason.message);
              this.handleError_('intent handler failed: %s', reason.message);
              this.lastErrorMessage_ = reason.message;
              return false;
            });
        } else {
          // Handle functions
          return true;
        }
        return true;
      }
    }
    this.handleError_('no matching intent handler for: ' + intent);
    return false;
  }

  /**
   * Utility function to detect SSML markup.
   *
   * @param {string} text The text to be checked.
   * @return {boolean} true if text is SSML markup.
   * @private
   */
  isSsml_ (text) {
    debug('isSsml_: text=%s', text);
    if (!text) {
      this.handleError_('text can NOT be empty.');
      return false;
    }
    return /^<speak\b[^>]*>([^]*?)<\/speak>$/gi.test(text);
  }

  /**
   * Utility function to detect incoming request format.
   *
   * @return {boolean} true if request is in Proto3 format.
   * @private
   */
  isProto3_ () {
    debug('isProto3_');
    return this.actionsApiVersion_ !== null &&
      parseInt(this.actionsApiVersion_, 10) >= ACTIONS_CONVERSATION_API_VERSION_TWO;
  }

  /**
   * Utility function to handle error messages.
   *
   * @param {string} text The error message.
   * @private
   */
  handleError_ (text) {
    debug('handleError_: text=%s', text);
    if (!text) {
      error('Missing text');
      return;
    }
    // Log error
    error.apply(text, Array.prototype.slice.call(arguments, 1));
    // Tell action to say error
    if (this.responded_) {
      return;
    }
    if (this.response_) {
      // Don't call other methods; just do directly
      this.response_.status(RESPONSE_CODE_BAD_REQUEST).send(API_ERROR_MESSAGE_PREFIX + text);
      this.responded_ = true;
    }
  }

  /**
   * Utility method to send an HTTP response.
   *
   * @param {string} response The JSON response.
   * @param {string} respnseCode The HTTP response code.
   * @return {Object} HTTP response.
   * @private
   */
  doResponse_ (response, responseCode) {
    debug('doResponse_: response=%s, responseCode=%d', JSON.stringify(response), responseCode);
    if (this.responded_) {
      return;
    }
    if (!response) {
      this.handleError_('Response can NOT be empty.');
      return null;
    }
    let code = RESPONSE_CODE_OK;
    if (responseCode) {
      code = responseCode;
    }
    if (this.apiVersion_ !== null) {
      this.response_.append(CONVERSATION_API_VERSION_HEADER, this.apiVersion_);
    }
    this.response_.append(HTTP_CONTENT_TYPE_HEADER, HTTP_CONTENT_TYPE_JSON);
    // If request was in Proto2 format, convert response to Proto2
    if (!this.isProto3_()) {
      if (response.data) {
        response.data = transformToSnakeCase(response.data);
      } else {
        response = transformToSnakeCase(response);
      }
    }
    debug('Response %s', JSON.stringify(response));
    const httpResponse = this.response_.status(code).send(response);
    this.responded_ = true;
    return httpResponse;
  }

  /**
   * Extract session data from the incoming JSON request.
   *
   * Used in subclasses for Actions SDK and API.AI.
   * @private
   */
  extractData_ () {
    debug('extractData_');
    this.data = {};
  }

  /**
   * Uses a PermissionsValueSpec object to construct and send a
   * permissions request to user.
   *
   * Used in subclasses for Actions SDK and API.AI.
   * @return {Object} HTTP response.
   * @private
   */
  fulfillPermissionsRequest_ () {
    debug('fulfillPermissionsRequest_');
    return {};
  }

  /**
   * Helper to build prompts from SSML's.
   *
   * @param {Array<string>} ssmls Array of ssml.
   * @return {Array<Object>} Array of SpeechResponse objects.
   * @private
   */
  buildPromptsFromSsmlHelper_ (ssmls) {
    debug('buildPromptsFromSsmlHelper_: ssmls=%s', ssmls);
    const prompts = [];
    for (let i = 0; i < ssmls.length; i++) {
      const prompt = {
        ssml: ssmls[i]
      };
      prompts.push(prompt);
    }
    return prompts;
  }

  /**
   * Helper to build prompts from plain texts.
   *
   * @param {Array<string>} plainTexts Array of plain text to speech.
   * @return {Array<Object>} Array of SpeechResponse objects.
   * @private
   */
  buildPromptsFromPlainTextHelper_ (plainTexts) {
    debug('buildPromptsFromPlainTextHelper_: plainTexts=%s', plainTexts);
    const prompts = [];
    for (let i = 0; i < plainTexts.length; i++) {
      const prompt = {
        textToSpeech: plainTexts[i]
      };
      prompts.push(prompt);
    }
    return prompts;
  }

  /**
   * Helper to build SimpleResponse from speech and display text.
   *
   * @param {string|SimpleResponse} response String to speak, or SimpleResponse.
   *     SSML allowed.
   * @param {string} response.speech If using SimpleResponse, speech to be spoken
   *     to user.
   * @param {string=} response.displayText If using SimpleResponse, text to be shown
   *     to user.
   * @return {Object} Appropriate SimpleResponse object.
   * @private
   */
  buildSimpleResponseHelper_ (response) {
    debug('buildSimpleResponseHelper_: response=%s', JSON.stringify(response));
    let simpleResponseObj = {};
    if (typeof response === 'string') {
      simpleResponseObj = this.isSsml_(response)
        ? { ssml: response } : { textToSpeech: response };
    } else if (response.speech) {
      simpleResponseObj = this.isSsml_(response.speech)
        ? { ssml: response.speech } : { textToSpeech: response.speech };
      simpleResponseObj.displayText = response.displayText;
    } else {
      this.handleError_('SimpleResponse requires a speech parameter.');
      return null;
    }
    return simpleResponseObj;
  }
};

/**
 * Utility class for representing intents by name.
 *
 * @private
 */
const Intent = class {
  constructor (name) {
    this.name_ = name;
  }

  getName () {
    return this.name_;
  }
};

/**
 * Utility class for representing states by name.
 *
 * @private
 */
const State = class {
  constructor (name) {
    this.name_ = name;
  }

  getName () {
    return this.name_;
  }
};

module.exports = {
  Action: Action,
  State: State
};

